/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

import java.util.List;
import java.util.ArrayList;

/**
   Assemble a {@link Model}'s {@link List} of {@link Primitive}s
   into a {@link List} containing only {@link LineSegment},
   {@link Points} and {@link Triangle} {@link Primitive}s.
<p>
   In order to clip a {@link Primitive}, we need to convert the
   {@link Primitive} into individual {@link LineSegment}s or
   {@link Triangle}s, and then clip each one separately.
*/
public class PrimitiveAssembly
{
   /**
      Assemble a {@link Model}'s {@link List} of {@link Primitive}s into a
      {@link List} containing only {@link LineSegment}, {@link Points}
      and {@link Triangle} {@link Prinitive}s.

      @param model  {@code Model} whose {@link Primitive}s need to be assembled
   */
   public static void assemble(Model model)
   {
      List<Primitive> result = new ArrayList<>();

      for (Primitive p : model.primitiveList)
      {
         if (p instanceof LineStrip)
         {
            // Convert the LineStrip into individual LineSegments.
            for (int i = 0; i < p.vIndexList.size() - 1; ++i)
            {
               result.add(new LineSegment(p.vIndexList.get(i),
                                          p.vIndexList.get(i+1),
                                          p.cIndexList.get(i),
                                          p.cIndexList.get(i+1)));
            }
         }
         else if (p instanceof LineLoop || p instanceof Face)
         {
            // Convert the LineLoop or Face into individual LineSegments.
            for (int i = 0; i < p.vIndexList.size() - 1; ++i)
            {
               result.add(new LineSegment(p.vIndexList.get(i),
                                          p.vIndexList.get(i+1),
                                          p.cIndexList.get(i),
                                          p.cIndexList.get(i+1)));
            }
            // Close the LineLoop or Face into a loop of line segments.
            result.add(new LineSegment(p.vIndexList.get(p.vIndexList.size() - 1),
                                       p.vIndexList.get(0),
                                       p.cIndexList.get(p.vIndexList.size() - 1),
                                       p.cIndexList.get(0)));
         }
         else if (p instanceof LineFan)
         {
            // Convert the LineFan into individual LineSegments.
            for (int i = 1; i < p.vIndexList.size(); ++i)
            {
               result.add(new LineSegment(p.vIndexList.get(0),
                                          p.vIndexList.get(i),
                                          p.cIndexList.get(0),
                                          p.cIndexList.get(i)));
            }
         }
         else if (p instanceof Lines)
         {
            // Convert the Lines into individual LineSegments.
            for (int i = 0; i < p.vIndexList.size(); i += 2)
            {
               result.add(new LineSegment(p.vIndexList.get(i),
                                          p.vIndexList.get(i + 1),
                                          p.cIndexList.get(i),
                                          p.cIndexList.get(i + 1)));
            }
         }
         else if (p instanceof TriangleStrip)
         {
            // Convert the TriangleStrip into individual Triangles.
            int c = 0;
            for (int i = 0; i < p.vIndexList.size() - 2; ++i)
            {
               int a = 1 + c; // a is either 1 or 2 (since c is 0 or 1)
               int b = 2 - c; // b is either 2 or 1 (since c is 0 or 1)
               Triangle newTri = new Triangle(
                                            p.vIndexList.get(i + 0),
                                            p.vIndexList.get(i + a),
                                            p.vIndexList.get(i + b),
                                            p.cIndexList.get(i + 0),
                                            p.cIndexList.get(i + a),
                                            p.cIndexList.get(i + b));
               TriangleStrip p2 = (TriangleStrip)p;
               newTri.setBackFaceColorIndex(p2.cIndexList2.get(i + 0),
                                            p2.cIndexList2.get(i + a),
                                            p2.cIndexList2.get(i + b));
               result.add(newTri);
               c = 1 - c;  // c alternates between 0 and 1
            }
         }
         else if (p instanceof TriangleFan)
         {
            // Convert the TriangleFan into individual Triangles.
            for (int i = 1; i < p.vIndexList.size() - 1; ++i)
            {
               Triangle newTri = new Triangle(
                                            p.vIndexList.get(0),
                                            p.vIndexList.get(i),
                                            p.vIndexList.get(i + 1),
                                            p.cIndexList.get(0),
                                            p.cIndexList.get(i),
                                            p.cIndexList.get(i + 1));
               TriangleFan p2 = (TriangleFan)p;
               newTri.setBackFaceColorIndex(p2.cIndexList2.get(0),
                                            p2.cIndexList2.get(i),
                                            p2.cIndexList2.get(i + 1));
               result.add(newTri);
            }
         }
         else if (p instanceof Triangles)
         {
            // Convert the Triangles into individual Triangles.
            for (int i = 0; i < p.vIndexList.size() - 2; i += 3)
            {
               Triangle newTri = new Triangle(
                                            p.vIndexList.get(i + 0),
                                            p.vIndexList.get(i + 1),
                                            p.vIndexList.get(i + 2),
                                            p.cIndexList.get(i + 0),
                                            p.cIndexList.get(i + 1),
                                            p.cIndexList.get(i + 2));
               Triangles p2 = (Triangles)p;
               newTri.setBackFaceColorIndex(p2.cIndexList2.get(i + 0),
                                            p2.cIndexList2.get(i + 1),
                                            p2.cIndexList2.get(i + 2));
               result.add(newTri);
            }
         }
         else if (p instanceof LineSegment)
         {
            result.add(p);
         }
         else if (p instanceof Triangle)
         {
            result.add(p);
         }
         else if (p instanceof Points)
         {
            result.add(p);
         }
         else if (p instanceof Point)
         {
            // Convert the Point object into a Points object.
            Points points = new Points();
            points.radius = ((Point)p).radius;
            points.addIndices(p.vIndexList.get(0),
                              p.cIndexList.get(0));
            result.add(points);
         }
      }

      model.primitiveList = result;
   }
}
