/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.*;
import  renderer.framebuffer.*;

import java.awt.Color;

/**
   Rasterize a clipped {@link LineSegment} into shaded pixels in
   the {@link FrameBuffer}'s viewport and (optionally) anti-alias
   the line at the same time.
<p>
   This pipeline stage takes a clipped {@link LineSegment} with
   vertices in the {@link Camera}'s view rectangle and rasterizezs
   the line segment into pixels in the {@link FrameBuffer}'s viewport.
   This rasterizer will linearly interpolate a color from the line
   segment's two endpoints to each rasterized pixel in the line segment.
<p>
   This rasterization algorithm is based on
<pre>
     "Fundamentals of Computer Graphics", 3rd Edition,
      by Peter Shirley, pages 163-165.
</pre>
<p>
   This is a fairly simplistic anti-aliasing algorithm.
*/
public class RasterizeLine
{
   public static boolean debug;

   /**
      Rasterize a clipped {@link LineSegment} into antialiased, shaded pixels
      in the {@link FrameBuffer}'s viewport.

      @param model  {@link Model} that the {@link LineSegment} {@code ls} comes from
      @param ls     {@link LineSegment} to rasterize into the {@link FrameBuffer}
      @param fb     {@link FrameBuffer} to hold rasterized pixels
   */
   public static void rasterize(Model model, LineSegment ls, FrameBuffer fb)
   {
      debug = Rasterize.debug; // update the debug status

      // Get the viewport's background color.
      Color bg = fb.getBgColorVP();

      // Make local copies of several values.
      int w = fb.getWidthVP();
      int h = fb.getHeightVP();

      Vertex v0 = model.vertexList.get( ls.vIndexList.get(0) );
      Vertex v1 = model.vertexList.get( ls.vIndexList.get(1) );

      float[] c0 = model.colorList.get( ls.cIndexList.get(0) ).getRGBComponents(null);
      float[] c1 = model.colorList.get( ls.cIndexList.get(1) ).getRGBComponents(null);
      float r0 = c0[0],  g0 = c0[1],  b0 = c0[2];
      float r1 = c1[0],  g1 = c1[1],  b1 = c1[2];

      // Transform each vertex to the "pixel plane" coordinate system.
      double x0 = 0.5 + w/2.001 * (v0.x + 1); // x_pp = 0.5 + w/2 * (x_p+1)
      double y0 = 0.5 + h/2.001 * (v0.y + 1); // y_pp = 0.5 + h/2 * (y_p+1)
      double x1 = 0.5 + w/2.001 * (v1.x + 1);
      double y1 = 0.5 + h/2.001 * (v1.y + 1);
      // NOTE: Notice the 2.001 fudge factor in the last two equations.
      // This is explained on page 142 of
      //    "Jim Blinn's Corner: A Trip Down The Graphics Pipeline"
      //     by Jim Blinn, 1996, Morgan Kaufmann Publishers.

      // Round each vertex to the nearest logical pixel.
      // This makes the algorithm a lot simpler, but it can
      // cause a slight, but noticeable, shift of the line segment.
      x0 = Math.round(x0);
      y0 = Math.round(y0);
      x1 = Math.round(x1);
      y1 = Math.round(y1);

      // Get the z coordinates.
      double z0 = v0.z;
      double z1 = v1.z;

      // Rasterize a degenerate line segment (a line segment
      // that projected onto a point) as a single pixel.
      if ( (x0 == x1) && (y0 == y1) )
      {
         // Get the z-buffer value currently at this pixel of the viewport.
         double zBuff = fb.getDepthVP((int)x0 - 1, h - (int)y0);

         // Determine which endpoint of the line segment is in front, and
         // if that endpoint is in front of what is in the framebuffer,
         // then overwrite the data in the framebuffer.
         if ( (z0 >= z1) && (z0 > zBuff) )
         {
            fb.setPixelVP((int)x0 - 1, h - (int)y0, new Color(r0, g0, b0));
            fb.setDepthVP((int)x0 - 1, h - (int)y0, z0);
         }
         else if (z1 > zBuff)
         {
            fb.setPixelVP((int)x1 - 1, h - (int)y1, new Color(r1, g1, b1));
            fb.setDepthVP((int)x1 - 1, h - (int)y1, z1);
         }

         // log interesting information to standard output
         if (debug) logPixel(w, h, (int)x0, y0, r0, g0, b0);

         return;
      }

      // If abs(slope) > 1, then transpose the line so that
      // the transposed line has slope < 1. Remember that the
      // line has been transposed.
      boolean transposedLine = false;
      if (Math.abs(y1 - y0) > Math.abs(x1 - x0))
      {  // swap x0 with y0 and swap x1 with y1
         double temp0 = x0;
         x0 = y0;
         y0 = temp0;
         double temp1 = x1;
         x1 = y1;
         y1 = temp1;
         transposedLine = true;
      }

      if (x1 < x0) // we want to rasterize from left-to-right
      {  // so, if necessary, swap (x0, y0, z0) with (x1, y1, z1)
         double tempX = x0;
         x0 = x1;
         x1 = tempX;
         double tempY = y0;
         y0 = y1;
         y1 = tempY;
         double tempZ = z0;
         z0 = z1;
         z1 = tempZ;
         // swap the colors too
         float tempR = r0;
         r0 = r1;
         r1 = tempR;
         float tempG = g0;
         g0 = g1;
         g1 = tempG;
         float tempB = b0;
         b0 = b1;
         b1 = tempB;
      }

      // Compute this line segment's slopes.
      double      m = (y1 - y0) / (x1 - x0);
      double slopeZ = (z1 - z0) / (x1 - x0);
      double slopeR = (r1 - r0) / (x1 - x0);
      double slopeG = (g1 - g0) / (x1 - x0);
      double slopeB = (b1 - b0) / (x1 - x0);

      // Rasterize this line segment, along the x-axis, from left-to-right.
      // In the following loop, as x moves across the logical
      // horizontal pixels, we will compute a y value for each x.
      double y = y0;
      for (int x = (int)x0; x < (int)x1; x += 1, y += m)
      {
         // Interpolate this pixel's depth between the two endpoint's depths.
         double z = z0 + slopeZ*(x - x0);

         // Get the z-buffer value currently at this pixel of the viewport.
         double zBuff = -Double.MAX_VALUE;
         if ( ! transposedLine )
         {
            zBuff = fb.getDepthVP(x - 1, h - (int)Math.round(y));
         }
         else // a transposed line
         {
            zBuff = fb.getDepthVP((int)Math.round(y) - 1, h - x);
         }

         // If the current fragment is in front of what is in the framebuffer,
         // then overwrite the data in the framebuffer.
         if (z > zBuff)
         {
            // Interpolate this pixel's color between the two endpoint's colors.
            float r = (float)Math.abs(r0 + slopeR*(x - x0));
            float g = (float)Math.abs(g0 + slopeG*(x - x0));
            float b = (float)Math.abs(b0 + slopeB*(x - x0));
            // We need the Math.abs() because otherwise, we sometimes get -0.0.

            if (Pipeline.doAntialiasing)
            {
               // y must be between two vertical logical pixel coordinates.
               // Let y_low and y_hi be the vertical logical pixel coordinates
               // that bracket around y.
               int y_low = (int)y;                  // the integer part of y
               int y_hi  = (y < h) ? y_low + 1 : h; // test for the top edge
               // Let weight be the fractional part of y. We will use
               // weight to determine how much emphasis to place on
               // each of the two pixels that bracket y.
               float weight = (float)(y - y_low);

               // Interpolate colors for the low and high pixels.
               // The smaller weight is, the closer y is to the lower
               // pixel, so we give the lower pixel more emphasis when
               // weight is small.
               float r_low = (float)((1 - weight) * r + weight * (bg.getRed()/255.0));
               float g_low = (float)((1 - weight) * g + weight * (bg.getGreen()/255.0));
               float b_low = (float)((1 - weight) * b + weight * (bg.getBlue()/255.0));
               float r_hi  = (float)(weight * r + (1 - weight) * (bg.getRed()/255.0));
               float g_hi  = (float)(weight * g + (1 - weight) * (bg.getGreen()/255.0));
               float b_hi  = (float)(weight * b + (1 - weight) * (bg.getBlue()/255.0));

               // Set this (antialiased) pixel in the framebuffer.
               if ( ! transposedLine )
               {
                  fb.setPixelVP(x - 1, h - y_low, new Color(r_low, g_low, b_low));
                  fb.setPixelVP(x - 1, h - y_hi,  new Color(r_hi,  g_hi,  b_hi));
                  fb.setDepthVP(x - 1, h - y_low, z);
                  fb.setDepthVP(x - 1, h - y_hi,  z);

                  // log interesting information to standard output
                  if (debug) logPixel(w, h, x, y, r_low, g_low, b_low, r_hi, g_hi, b_hi);
               }
               else // a transposed line
               {
                  fb.setPixelVP(y_low - 1, h - x, new Color(r_low, g_low, b_low));
                  fb.setPixelVP(y_hi  - 1, h - x, new Color(r_hi,  g_hi,  b_hi));
                  fb.setDepthVP(y_low - 1, h - x, z);
                  fb.setDepthVP(y_hi  - 1, h - x, z);

                  // log interesting information to standard output
                  if (debug) logPixel(w, h, x, y, r_low, g_low, b_low, r_hi, g_hi, b_hi);
               }
            }
            else // no antialiasing
            {
               // The value of y will almost always be between
               // two vertical pixel coordinates. By rounding off
               // the value of y, we are choosing the nearest logical
               // vertical pixel coordinate.
               if ( ! transposedLine )
               {
                  fb.setPixelVP(x - 1, h - (int)Math.round(y), new Color(r, g, b));
                  fb.setDepthVP(x - 1, h - (int)Math.round(y), z);

                  // log interesting information to standard output
                  if (debug) logPixel(w, h, x, y, r, g, b);
               }
               else // a transposed line
               {
                  fb.setPixelVP((int)Math.round(y) - 1, h - x, new Color(r, g, b));
                  fb.setDepthVP((int)Math.round(y) - 1, h - x, z);

                  // log interesting information to standard output
                  if (debug) logPixel(w, h, y, x, r, g, b);
               }
            }
         }
         // Advance (x,y) to the next pixel (delta_x is 1, so delta_y is m).
      }
      // Set the pixel for the (x1,y1) endpoint.
      // We do this separately to avoid roundoff errors.
      if ( ! transposedLine )
      {
         double zBuff = fb.getDepthVP((int)x1 - 1, h - (int)y1);
         if (z1 > zBuff)
         {
            fb.setPixelVP((int)x1 - 1, h - (int)y1, new Color(r1, g1, b1));
            fb.setDepthVP((int)x1 - 1, h - (int)y1, z1);
         }
      }
      else // a transposed line
      {
         double zBuff = fb.getDepthVP((int)y1 - 1, h - (int)x1);
         if (z1 > zBuff)
         {
            fb.setPixelVP((int)y1 - 1, h - (int)x1, new Color(r1, g1, b1));
            fb.setDepthVP((int)y1 - 1, h - (int)x1, z1);
         }
      }
   }


   private static void logPixel(int w, int h, int x, double y,
                                float r, float g, float b)
   {
      System.err.printf(
         "[w=%d, h=%d]  x=%d, y=%f, r=%f, g=%f, b=%f\n", w, h, x, y, r, g, b);
   }

   private static void logPixel(int w, int h, double x, int y,
                                float r, float g, float b)
   {
      System.err.printf(
         "[w=%d, h=%d]  x=%f, y=%d, r=%f, g=%f, b=%f\n", w, h, x, y, r, g, b);
   }

   private static void logPixel(int w, int h, int x, double y,
                                float r1, float g1, float b1,
                                float r2, float g2, float b2)
   {
      System.err.printf(
         "[w=%d, h=%d]  x=%d, y=%f, r=%f, g=%f, b=%f\n", w, h, x, y, r1, g1, b1);
      System.err.printf(
         "              x=%d, y=%f, r=%f, g=%f, b=%f\n",       x, y, r2, g2, b2);
   }

   private static void logPixel(int w, int h, double x, int y,
                                float r1, float g1, float b1,
                                float r2, float g2, float b2)
   {
      System.err.printf(
         "[w=%d, h=%d]  x=%f, y=%d, r=%f, g=%f, b=%f\n", w, h, x, y, r1, g1, b1);
      System.err.printf(
         "              x=%f, y=%d, r=%f, g=%f, b=%f\n",       x, y, r2, g2, b2);
   }
}
