/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.*;
import  renderer.framebuffer.*;

import java.awt.Color;

/**
   Rasterize a clipped {@link Points} into pixels in the
   {@link FrameBuffer}'s viewport.
*/
public class RasterizePoints
{
   public static boolean debug;

   /**
      Rasterize a clipped {@link Points} into pixels in the
      {@link FrameBuffer}'s viewport.

      @param model  {@link Model} that the {@link Points} {@code pts} comes from
      @param pts    {@code Points} to rasterize into the {@code FrameBuffer}
      @param fb     {@code FrameBuffer} to hold rasterized pixels
   */
   public static void rasterize(Model model, Primitive pts, FrameBuffer fb)
   {
      debug = Rasterize.debug; // update the debug status

      int w = fb.getWidthVP();
      int h = fb.getHeightVP();

      for (int i = 0; i < pts.vIndexList.size(); ++i)
      {
         Vertex  v = model.vertexList.get(pts.vIndexList.get(i));
         float[] c = model.colorList.get(pts.cIndexList.get(i)).getRGBComponents(null);

         // Transform each vertex to the "pixel plane" coordinate system.
         double x = 0.5 + w/2.001 * (v.x + 1); // x_pp = 0.5 + w/2 * (x_p+1)
         double y = 0.5 + h/2.001 * (v.y + 1); // y_pp = 0.5 + h/2 * (y_p+1)
         // NOTE: Notice the 2.001 fudge factor in the last two equations.
         // This is explained on page 142 of
         //    "Jim Blinn's Corner: A Trip Down The Graphics Pipeline"
         //     by Jim Blinn, 1996, Morgan Kaufmann Publishers.

         // Round the point's coordinates to the nearest logical pixel.
         x = Math.round( x );
         y = Math.round( y );

         // Get this points's depth.
         double z = v.z;

         // Get the z-buffer value currently at this pixel of the viewport.
         double zBuff = fb.getDepthVP((int)x - 1, h - (int)y);

         // If the current fragment is in front of what is in the framebuffer,
         // then overwrite the data in the framebuffer.
         if (z > zBuff)
         {
            // Get the color of the point.
            double r = c[0];
            double g = c[1];
            double b = c[2];

            // Get this points's radius.
            int radius = ((Points)pts).radius;

            for (int y_ = (int)y - radius; y_ <= (int)y + radius; ++y_)
            {
               for (int x_ = (int)x - radius; x_ <= (int)x + radius; ++x_)
               {
                  if (x_ >= 0 && x_ < w && y_ > 0 && y_ < h) // clipping test
                  {
                     fb.setPixelVP( x_ - 1,
                                    h - y_,
                                    new Color((float)r, (float)g, (float)b) );

                     // log interesting information to standard output
                     if (debug) logPixel(w, h, x_, y_, r, g, b);
                  }
               }
            }
         }
      }
   }

   private static void logPixel(int w, int h, double x, double y, double r, double g, double b)
   {
      System.err.printf("[w=%d, h=%d]  x=%f, y=%f, r=%f, g=%f, b=%f\n", w, h, x, y, r, g, b);
   }
}
