/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.*;
import  renderer.framebuffer.*;

import java.awt.Color;

/**
   Rasterize a clipped {@link Triangle} into pixels in the
   {@link FrameBuffer}'s viewport.
<p>
   This rasterization algorithm is based on
<pre>
     "Fundamentals of Computer Graphics", 3rd Edition,
      by Peter Shirley, pages 163-165.
</pre>
*/
public class RasterizeTriangle
{
   /**
      Rasterize a clipped {@link Triangle} into pixels in the
      {@link FrameBuffer}'s viewport.

      @param model  {@link Model} that the {@link Triangle} {@code tri} comes from
      @param tri    {@code Triangle} to rasterize into the {@code FrameBuffer}
      @param fb     {@code FrameBuffer} to hold rasterized pixels
   */
   public static void rasterize(Model model, Triangle tri, FrameBuffer fb)
   {
      boolean debug = Rasterize.debug;

      // Make local copies of several values.
      int w = fb.getWidthVP();
      int h = fb.getHeightVP();

      int vIndex0 = tri.vIndexList.get(0);
      int vIndex1 = tri.vIndexList.get(1);
      int vIndex2 = tri.vIndexList.get(2);

      int cIndex0 = tri.cIndexList.get(0);
      int cIndex1 = tri.cIndexList.get(1);
      int cIndex2 = tri.cIndexList.get(2);

      Vertex v0 = model.vertexList.get(vIndex0);
      Vertex v1 = model.vertexList.get(vIndex1);
      Vertex v2 = model.vertexList.get(vIndex2);

      float[] c0 = model.colorList.get(cIndex0).getRGBComponents(null);
      float[] c1 = model.colorList.get(cIndex1).getRGBComponents(null);
      float[] c2 = model.colorList.get(cIndex2).getRGBComponents(null);

      // Get the coordinates for the three vertices.
      double x0 = v0.x,  y0 = v0.y,  z0 = v0.z;
      double x1 = v1.x,  y1 = v1.y,  z1 = v1.z;
      double x2 = v2.x,  y2 = v2.y,  z2 = v2.z;

      // Get the colors for the three vertices.
      float r0 = c0[0],  g0 = c0[1],  b0 = c0[2];
      float r1 = c1[0],  g1 = c1[1],  b1 = c1[2];
      float r2 = c2[0],  g2 = c2[1],  b2 = c2[2];

      // Rasterize a degenerate triangle (a triangle that
      // projected onto a line) as three line segments.
      double area = (x1-x0)*(y2-y0) - (y1-y0)*(x2-x0);
      if ( 0.0001 > Math.abs(area) )
      {
         RasterizeLine.rasterize(model, new LineSegment(vIndex0, vIndex1, cIndex0, cIndex1), fb);
         RasterizeLine.rasterize(model, new LineSegment(vIndex1, vIndex2, cIndex1, cIndex2), fb);
         RasterizeLine.rasterize(model, new LineSegment(vIndex2, vIndex0, cIndex2, cIndex0), fb);
         return;
      }

      // Transform each vertex to the "pixel plane" coordinate system.
      x0 = 0.5 + w/2.001 * (x0 + 1); // x_pp = 0.5 + w/2 * (x_p+1)
      y0 = 0.5 + h/2.001 * (y0 + 1); // y_pp = 0.5 + h/2 * (y_p+1)
      x1 = 0.5 + w/2.001 * (x1 + 1);
      y1 = 0.5 + h/2.001 * (y1 + 1);
      x2 = 0.5 + w/2.001 * (x2 + 1);
      y2 = 0.5 + h/2.001 * (y2 + 1);
      // NOTE: Notice the 2.001 fudge factor in the last two equations.
      // This is explained on page 142 of
      //    "Jim Blinn's Corner: A Trip Down The Graphics Pipeline"
      //     by Jim Blinn, 1996, Morgan Kaufmann Publishers.

      // Recall that this triangle's viewport coordinates are in the intervals
      //     x in (-0.5, vpWidth  - 0.5)
      //     y in (-0.5, vpHeight - 0.5)
      // We want to "sample" the triangle (in viewport coordinates)
      // at the pixel locations with integer coordinates between
      //     0 and vpWidth - 1
      //     0 and vpHeight - 1
      // Our first step is to find a "pixel bounding box" for the triangle.
      // Each vertex of the triangle is rounded to integer coordinates, and
      // then we find the left-most, right-most, highest, and lowest integer
      // coordinates that bound the triangle's vertices.

      // Find the greatest integer less than or equal to x0, x1, and x2.
      int xMin = (int)x0;
      if (x1 < xMin) xMin = (int)x1;
      if (x2 < xMin) xMin = (int)x2;
      if (xMin == -1) xMin = 0;

      // Find the greatest integer less than or equal to y0, y1, and y2.
      int yMin = (int)y0;
      if (y1 < yMin) yMin = (int)y1;
      if (y2 < yMin) yMin = (int)y2;
      if (yMin == -1) yMin = 0;

      // Find the least integer greater than or equal to x0, x1, and x2.
      int xMax = (int)Math.ceil(x0);
      if (x1 > xMax) xMax = (int)Math.ceil(x1);
      if (x2 > xMax) xMax = (int)Math.ceil(x2);
      if (xMax == fb.getWidthVP()) --xMax;

      // Find the least integer greater than or equal to y0, y1, and y2.
      int yMax = (int)Math.ceil(y0);
      if (y1 > yMax) yMax = (int)Math.ceil(y1);
      if (y2 > yMax) yMax = (int)Math.ceil(y2);
      if (yMax == fb.getHeightVP()) --yMax;
      if (debug) System.err.printf("xMin = %d, xMax = %d, yMin = %d, yMax = %d\n", xMin,xMax,yMin,yMax);

    //for (int y = yMin;   y <  yMax; ++y) // bottom to top
      for (int y = yMax-1; y >= yMin; --y) // top to bottom
      {
         for (int x = xMin; x < xMax; ++x)  /// left to right
         {
            //  f01(x,y) = (y0-y1)*(x) + (x1-x0)*(y) + x0*y1 - x1*y0
            //  f12(x,y) = (y1-y2)*(x) + (x2-x1)*(y) + x1*y2 - x2*y1
            //  f20(x,y) = (y2-y0)*(x) + (x0-x2)*(y) + x2*y0 - x0*y2

            //  alpha = f12(x,y) / f12(x0,y0)
            //  beta  = f20(x,y) / f20(x1,y1)
            //  gamma = f01(x,y) / f01(x2,y2)

            double alpha = ( (y1-y2)*(x)  + (x2-x1)*(y)  + x1*y2 - x2*y1 )
                         / ( (y1-y2)*(x0) + (x2-x1)*(y0) + x1*y2 - x2*y1 );
            double beta  = ( (y2-y0)*(x)  + (x0-x2)*(y)  + x2*y0 - x0*y2 )
                         / ( (y2-y0)*(x1) + (x0-x2)*(y1) + x2*y0 - x0*y2 );
            double gamma = ( (y0-y1)*(x)  + (x1-x0)*(y)  + x0*y1 - x1*y0 )
                         / ( (y0-y1)*(x2) + (x1-x0)*(y2) + x0*y1 - x1*y0 );

            //System.err.printf("alpha = %f, beta = %f, gamma = %f\n", alpha,beta,gamma);

            // Create a new fragment for the FrameBuffer's viewport.
            if (alpha >= 0 && beta >= 0 && gamma >= 0)
            {
               // Interpolate z-coordinate from the triangle's vertices to this fragment.
               double zCoord = (alpha * z0) + (beta * z1) + (gamma * z2);

               // Get the z-buffer value currently at this pixel of the viewport.
               double zBuff = fb.getDepthVP(x - 1, h - y);

               // If the current fragment is in front of the pixel in the framebuffer,
               // then overwrite the pixel in the framebuffer with this fragment.
               if (zCoord > zBuff)
               {
                  // Interpolate color data from the triangle's vertices to this fragment.
                  float r = (float)( (alpha * r0) + (beta * r1) + (gamma * r2) );
                  float g = (float)( (alpha * g0) + (beta * g1) + (gamma * g2) );
                  float b = (float)( (alpha * b0) + (beta * b1) + (gamma * b2) );
                  //System.err.printf("r = %f, g = %f, b = %f\n", r, g, b);

                  fb.setPixelVP(x - 1, h - y, new Color(r, g, b), zCoord);

                  // log interesting information to standard output
                  if (debug) logPixel(w, h, x, y, zCoord, r, g, b);
               }
            }
         }
      }
   }


   private static void logPixel(int w, int h, int x, int y, double z,
                                float r, float g, float b)
   {
      System.out.printf(
         "[w=%d, h=%d]  x=%d, y=%d, z=%f, r=%f, g=%f, b=%f\n",
           w, h, x, y, z, r, g, b);
   }
}
