/*

*/

package renderer.scene;

/**
   This {@code Camera} data structure represents a camera that
   can be moved around and aimed within world coordinates.
<p>
   This camera has a {@link viewMatrix} which associates to this
   canera a location and an orientation in the world coordinate
   system. A Camera object is positioned and aimed within world
   coordinates using its {@link #viewTranslate} and
   {@link #viewRotate} methods.
<p>
   A camera's {@code viewTranslate()} and {@code viewRotate()}
   methods are used to position and orient the camera in world
   coordinates exactly as the {@link Model#modelTranslate} and
   {@link Model#modelRotate} methods are used to position and
   orient a model in world coordinates by setting the model's
   {@link Model#modelMatrix}.
<p>
   A camera's view matrix is the inverse of what you might expect
   it to be. If this camera is told to translate forward (in the
   positive x-direction) 5 units, then the view matrix will be a
   translation in the x-direction of -5 units. If this camera is
   told to look to the right (rotate around the y-axis) by say 20
   degrees, then the view matrix will be a rotation around by y-axis
   by -20 degrees.
<p>
   The reason for this is that a camera's view matrix is really a
   modelling matrix that should be applied to every model in a scene
   (in adition to each model's individual model matrix). When the camera moves forward by five units in the x-direction, every vertex in the scene becomes 5 units closer to the camera (the camera which is fixed at the origin and looking down the negative z-axis). So every vertex in the scene needs a modeling transformation that translates by -5 in the x-direction.



<p>
   This {@code Camera} also has a (@link normalizeMatrix} which
   associates to this camera a "view volume" that determines
   what part of space the camera "sees" when we use the camera
   to take a picture (that is, when we render a {@link Scene}).
<p>
   This {@code Camera} can "take a picture" two ways, using
   a perspective projection or a parallel (orthographic)
   projection. Each way of taking a picture has a different
   shape for its view volume. The data in this data structure
   determines the shape of each of the two view volumes.
<p>
   For the perspective projection, the view volume (in view
   coordinates!) is an infinitely long pyramid that is formed
   by the pyramid with its apex at the origin and its base in
   the plane {@code z = -near} with edges {@code x = left},
   {@code x = right}, {@code y = top}, and {@code y = bottom}.
   The perspective view volume's shape is set by the
   {@link projPerspective} methods.
<p>
   For the orthographic projection, the view volume (in view
   coordinates!) is an infinitely long rectangular cylinder
   parallel to the z-axis and with sides {@code x = left},
   {@code x = right}, {@code y = top}, and {@code y = bottom}
   (an infinite parallelepiped). The orthographic view volume's
   shape is set by the {@link projOrtho} method.
<p>
   When the graphics rendering {@link renderer.pipeline.Pipeline} uses
   this Camera to render a Scene, the renderer "sees" the geometry from
   the scene that is contained in this camera's view volume. (Notice
   that this means the orthographic camera will see geometry that
   is behind the camera. In fact, the perspective camera also sees
   geometry that is behind the camera.) The renderer's
   {@link renderer.pipeline.Clip} pipeline stage is responsible for
   making sure that the scene's geometry that is outside of this
   camera's view volume is not visible.
<p>
   The plane {@code z = -near} (in view coordinates) is the camera's
   image plane. The rectangle in the image plane with corners
   {@code (left, bottom, -near)} and {@code (right, top, -near)} is
   the camera's view rectangle. The view rectangle is like the film
   in a real camera, it is where the camera's image appears when you
   take a picture. The contents of the camera's view rectangle (after
   it gets "normalized" by the renderer's {@link pipeline.View2Camera}
   stage) is what gets rasterized, by the renderer's
   {@link pipeline.Viewport} and {@link pipeline.Rasterize} stages,
   into a {@link framebuffer.FrameBuffer}.
*/
public class Camera
{
   // Choose either perspective or parallel projection.
   public boolean perspective;

   public Matrix normalizeMatrix;
   public Matrix viewMatrix;
   public Matrix projMatrix;

   public double left;   // These numbers define the camera's view volume.
   public double right;  // These numbers are encoded into the normalization
   public double bottom; // matrix.
   public double top;
   public double n;
   public double f;


   /**
      The default {@code Camera} has the standard (normalized)
      perspective view volume and is located at the origin (of
      world coordinates), looking down the negative z-axis.
   */
   public Camera()
   {
      this.projPerspective();
      viewMatrix = Matrix.identity(); // identity matrix
   }


   /**
      Set up this {@code Camera}'s view volume as a perspective
      projection of the normalized view frustum extending along
      the z-axis from the near plane {@code z = -1} to the
      far plane {@code z = -3}.
   */
   public void projPerspective()
   {
      projPerspective(-1.0, +1.0, -1.0, +1.0, 1.0, 3.0);
   }


   /**
      Set up this {@code Camera}'s view volume as a perspective projection
      of a view frustun extending along the z-axis from the near plane to
      the far plane.

      @param left    left edge of view rectangle in the near plane
      @param right   right edge of view rectangle in the near plane
      @param bottom  bottom edge of view rectangle in the near plane
      @param top     top edge of view rectangle in the near plane
      @param near    distance from the camera to the near plane
      @param far     distance from the camera to the far plane
   */
   public void projPerspective(double left, double right, double bottom, double top, double near, double far)
   {
      this.left   = left;   // These numbers define the camera's view volume.
      this.right  = right;  // These numbers are encoded into the normalization
      this.bottom = bottom; // matrix.
      this.top    = top;
      this.n      = -near;
      this.f      = -far;

      this.normalizeMatrix = PerspectiveNormalizeMatrix.build(left, right, bottom, top, near);

      this.projMatrix = PerspectiveProjectionMatrix.build(near, far);

      this.perspective = true;
   }


   /**
      An alternative way to determine this {@code Camera}'s perspective
      view volume.
      <p>
      Here, the view volume is determined by a vertical "field of view"
      angle and an aspect ratio for the view rectangle in the near plane.

      @param fovy    angle in the y-direction subtended by the view rectangle in the near plane
      @param aspect  aspect ratio of the view rectangle in the near plane
      @param near    distance from the camera to the near plane
      @param far     distance from the camera to the far plane
   */
   public void projPerspective(double fovy, double aspect, double near, double far)
   {
      this.top    =  near * Math.tan((Math.PI/180.0)*fovy/2.0);
      this.bottom = -top;
      this.right  =  top * aspect;
      this.left   = -right;
      this.n      = -near;
      this.f      = -far;

      projPerspective(left, right, bottom, top, near, far);
   }


   /**
      Set up this {@code Camera}'s view volume as a parallel (orthographic)
      projection of the normalized view parallelepiped extending along the
      z-axis from the near plane {@code z = -1} to the far plane {@code z = -3}.
   */
   public void projOrtho()
   {
      projOrtho(-1.0, +1.0, -1.0, +1.0, 1.0, 3.0);
   }


   /**
      Set up this {@code Camera}'s view volume as a parallel (orthographic)
      projection of a view parallelepiped extending along the z-axis from
      the near plane to the far plane.

      @param left    left edge of view rectangle in the near plane
      @param right   right edge of view rectangle in the near plane
      @param bottom  bottom edge of view rectangle in the near plane
      @param top     top edge of view rectangle in the near plane
      @param near    distance from the camera to the near plane
      @param far     distance from the camera to the far plane
   */
   public void projOrtho(double left, double right, double bottom, double top, double near, double far)
   {
      this.left   = left;   // These numbers define the camera's view volume.
      this.right  = right;  // These numbers are encoded into the normalization
      this.bottom = bottom; // matrix.
      this.top    = top;
      this.n      = -near;
      this.f      = -far;

      this.normalizeMatrix = OrthographicNormalizeMatrix.build(left, right, bottom, top);

      this.projMatrix = OrthographicProjectionMatrix.build(near, far);

      this.perspective = false;
   }


   /**
      An alternative way to determine this {@code Camera}'s orthographic
      view volume.
      <p>
      Here, the view volume is determined by a vertical "field of view"
      angle and an aspect ratio for the view rectangle in the near plane.

      @param fovy    angle in the y-direction subtended by the view rectangle in the near plane
      @param aspect  aspect ratio of the view rectangle in the near plane
      @param near    distance from the camera to the near plane
      @param far     distance from the camera to the far plane
   */
   public void projOrtho(double fovy, double aspect, double near, double far)
   {
      this.top    =  near * Math.tan((Math.PI/180.0)*fovy/2.0);
      this.bottom = -top;
      this.right  =  top * aspect;
      this.left   = -right;

      projOrtho(left, right, bottom, top, near, far);
   }


   /**
      Set the location and orientation of this (@code Camera} in the world
      coordinate system.
   <p>
      Compare with
      <a href="https://www.opengl.org/sdk/docs/man2/xhtml/gluLookAt.xml" target="_top">
               https://www.opengl.org/sdk/docs/man2/xhtml/gluLookAt.xml</a>

      @param eyex     x-coordinate of the camera's location
      @param eyey     y-coordinate of the camera's location
      @param eyez     z-coordinate of the camera's location
      @param centerx  x-coordinate of the camera's look-at point
      @param centery  y-coordinate of the camera's look-at point
      @param centerz  z-coordinate of the camera's look-at point
      @param upx      x-component of the camera's up vector
      @param upy      y-component of the camera's up vector
      @param upz      z-component of the camera's up vector
   */
   public void viewLookAt(double eyex,    double eyey,    double eyez,
                          double centerx, double centery, double centerz,
                          double upx,     double upy,     double upz)
   {
      Vector F  = new Vector(centerx - eyex, centery - eyey, centerz - eyez);
      Vector UP = new Vector(upx, upy, upz);
      Vector f  = F.normalize();
      Vector up = UP.normalize();
      Vector s  = f.crossProduct(up);
      Vector u  = s.crossProduct(f);
      viewMatrix = Matrix.build(
                     new Vector(s.x, u.x, -f.x, 0.0),
                     new Vector(s.y, u.y, -f.y, 0.0),
                     new Vector(s.z, u.z, -f.z, 0.0),
                     new Vector(0.0, 0.0,  0.0, 1.0));

      viewMatrix = viewMatrix.times(Matrix.translate(-eyex, -eyey, -eyez));
   }


   /**
      Set this (@code Camera}'s view matrix to the identity {@link Matrix}.
   <p>
      This places the camera at the origin of world coordinates,
      looking down the negative z-axis.
   */
   public void view2Identity()
   {
      viewMatrix = Matrix.identity();
   }


   /**
      Translate this (@code Camera} in world coordinates by the amount of
      the given translation vector.
   <p>
      This means that we should left-multiply this camera's view matrix
      with a translation {@link Matrix} that is the inverse of the
      given translation.

      @param x  x-component of the Camera's translation vector
      @param y  y-component of the Camera's translation vector
      @param z  z-component of the Camera's translation vector
   */
   public void viewTranslate(double x, double y, double z)
   {
      // Notice that the order of the multiplication is the oppposite
      // of what we usually use. This is because the viewMatrix should
      // be the inverse of the matrix that would position the camera
      // in world coordinates. If A*B would be the position matrix,
      // its inverse (A*B)^(-1) = B^(-1) * A^(-1), so we see a reversal
      // in the order of multiplication.
      viewMatrix = (Matrix.translate(-x, -y, -z)).times(viewMatrix);
   }


   /**
      Rotate this {@code Camera} in world coordinates by the given angle
      around the given axis vector.
   <p>
      This means that we should left-multiply this {@code Camera}'s view
      matrix with a rotation {@link Matrix} that is the inverse of
      the given rotation.

      @param theta  angle, in degrees, to rotate the Camera by
      @param x      x-component of the axis vector for the Camera's rotation
      @param y      y-component of the axis vector for the Camera's rotation
      @param z      z-component of the axis vector for the Camera's rotation
   */
   public void viewRotate(double theta, double x, double y, double z)
   {
      // See the comment in the viewTranslate() method.
      viewMatrix = (Matrix.rotate(-theta, x, y, z)).times(viewMatrix);
   }


   /**
      Rotate this {@code Camera} in world coordinates by the given angle
      around the x-axis.
   <p>
      This means that we should left-multiply this {@code Camera}'s view
      matrix with a rotation {@link Matrix} that is the inverse of
      the given rotation.

      @param theta  angle, in degrees, to rotate the Camera by
   */
   public void viewRotateX(double theta)
   {
      // See the comment in the viewTranslate() method.
      viewMatrix = (Matrix.rotate(-theta, 1, 0, 0)).times(viewMatrix);
   }


   /**
      Rotate this {@code Camera} in world coordinates by the given angle
      around the y-axis.
   <p>
      This means that we should left-multiply this {@code Camera}'s view
      matrix with a rotation {@link Matrix} that is the inverse of
      the given rotation.

      @param theta  angle, in degrees, to rotate the Camera by
   */
   public void viewRotateY(double theta)
   {
      // See the comment in the viewTranslate() method.
      viewMatrix = (Matrix.rotate(-theta, 0, 1, 0)).times(viewMatrix);
   }


   /**
      Rotate this {@code Camera} in world coordinates by the given angle
      around the z-axis.
   <p>
      This means that we should left-multiply this {@code Camera}'s view
      matrix with a rotation {@link Matrix} that is the inverse of
      the given rotation.

      @param theta  angle, in degrees, to rotate the Camera by
   */
   public void viewRotateZ(double theta)
   {
      // See the comment in the viewTranslate() method.
      viewMatrix = (Matrix.rotate(-theta, 0, 0, 1)).times(viewMatrix);
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code Camera} object
   */
   public String toString()
   {
      double fovy = 2.0 * (180./Math.PI) * Math.atan(top/(-n));
      String result = "";
      result += "Camera: \n";
      result += "perspective = " + perspective + "\n";
      result += "left = "   + left + ", "
             +  "right = "  + right + "\n"
             +  "bottom = " + bottom + ", "
             +  "top = "    + top + "\n"
             +  "near = "   + -n + "\n"
             +  "far = "    + -f + "\n"
             +  "(fovy = " + fovy + ", aspect = " + right/top + ")\n"
             +  "Normalization Matrix\n"
             +  normalizeMatrix
             +  "View Matrix\n"
             +  viewMatrix
             +  "Projection Matrix\n"
             +  projMatrix;
      return result;
   }
}
