/*

*/

package renderer.scene;
import  renderer.scene.primitives.*;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;
import java.util.Random;

/**
   A {@code Model} data structure represents a distinct geometric object in
   a {@link Scene}. A {@code Model} data structure is mainly a {@link List}
   of {@link Vertex} objects, a list of {@link Primitive} objects, and a
   list of {@link Color} objects. {@link Primitive} objects represent
   "geometric primitives" that the model's geometry is build out of. There
   are nine kinds of geometric primitives, {@link Triangle}, {@link Face},
   {@link LineSegment}, {@link LineStrip}, {@link LineLoop}, {@link LineFan},
   {@link Lines}, {@link Point} and (@link Points}. Each {@link Primitive}
   object contains two lists of integers, one list made up of indices of
   {@link Vertex} objects from the {@code Model}'s vertex list, and the other
   list made up of indices of {@link Color} objects from the {@code Model}'s
   color list. Each {@link Vertex} object contains the coordinates, in the
   model's local coordinate system, for one point in the primitive. Each
   {@link Color} object contains the rgb values for a point in the primitive.
<p>
   A {@code Model} is also a recursive data structure. Every {@code Model}
   object contains a {@link List} of {@code Model} objects. The list of
   {@link Model}s lets us define what are called hierarchical, or nested,
   models. These are models that are made up of distinct parts and the
   distinct parts need to move both individually and also move together as
   a group. Each {@link Model} object nested inside of a {@code Model}
   contains its own {@link Matrix} which determines a local coordinate
   system for the vertices in that nested {@code Model} and for all the
   other {@code Model} objects that might be nested within that one.
<p>
   Each {@code Model}'s nested {@link Matrix} will place the nested
   {@link Model}'s {@link Model} within its parent's local coordinate
   system. Another way to say this is that the {@link Matrix} of a nested
   {@link Model} determines its {@link Model}'s location and orientation
   within the local coordinate system of the nested {@link Model}'s
   parent {@link Model}. In addition, the nested {@link Model}'s
   {@link Matrix} determines a new local coordinate system within which all
   the nested {@link Model}s below a {@link Model} will place their
   {@link Model}s


<p>
   See
<br> <a href="https://en.wikipedia.org/wiki/3D_modeling#Models" target="_top">
              https://en.wikipedia.org/wiki/3D_modeling#Models</a>
*/
public class Model
{
   public boolean doBackFaceCulling = true;
   public boolean frontFacingIsCCW  = true;
   public boolean facesHaveTwoSides = false;

   public List<Vertex> vertexList = new ArrayList<>();
   public List<Primitive> primitiveList = new ArrayList<>();
   public List<Color> colorList = new ArrayList<>();

   public Matrix nestedMatrix = Matrix.identity(); // identity matrix
   public List<Model> nestedModels = new ArrayList<>();

   public boolean visible;


   /**
      Construct an empty {@code Model} object.
   */
   public Model()
   {
      visible = true;
      doBackFaceCulling = true;
      frontFacingIsCCW = true;
      facesHaveTwoSides = false;
   }


   /**
      A "copy constructor". This constructor should make a deep copy
      of the given {@code Model}'s {@link Vertex} list, {@link Primitive}
      list, {@link Color} list, nested {@link Matrix}, and nested {@link Model} list.

      @param model  {@code Model} to make a copy of
   */
   public Model(Model model) // a "copy constructor"
   {
      super();

      this.visible = model.visible;
      this.doBackFaceCulling = model.doBackFaceCulling;
      this.frontFacingIsCCW = model.frontFacingIsCCW;
      this.facesHaveTwoSides = model.facesHaveTwoSides;

      for (Vertex v : model.vertexList)
      {
         this.vertexList.add(new Vertex(v)); // deep copy of each Vertex
      }
      for (Primitive p : model.primitiveList)
      {
         this.primitiveList.add(p.makeCopy()); // deep copy of each Primitive
      }
      for (Color c : model.colorList)
      {
         this.colorList.add(c); // Color objects are immutable
      }
      this.nestedMatrix = new Matrix(model.nestedMatrix);
      for (Model m : model.nestedModels)
      {
         this.nestedModels.add( new Model(m) ); // deep copy of each nested Model
      }
   }


   /**
      Add a {@link Vertex} (or vertices) to this {@code Model}'s
      {@link List} of vertices.

      @param vArray  array of {@link Vertex} objects to add to this {@code Model}
   */
   public void addVertex(Vertex... vArray)
   {
      for (Vertex v : vArray)
      {
         this.vertexList.add(new Vertex(v)); // NOTE: deep copy!
      }
   }


   /**
      Get a {@link Primitive} from this {@code Model}'s
      {@link List} of primitives.

      @param index  integer index of a {@link Primitive} from this {@code Model}
      @return the {@link Primitive} object at the given index
   */
   public Primitive getPrimitive(int index)
   {
      return this.primitiveList.get(index);
   }


   /**
      Add a {@link Primitive} (or primitives) to this {@code Model}'s
      {@link List} of primitives.

      @param pArray  array of {@link Primitive} objects to add to this {@code Model}
   */
   public void addPrimitive(Primitive... pArray)
   {
      for (Primitive p : pArray)
      {
         this.primitiveList.add(p);
      }
   }


   /**
      Add a {@link Color} (or colors) to this {@code Model}'s
      {@link List} of colors.

      @param cArray  array of {@link Color} objects to add to this {@code Model}
   */
   public void addColor(Color... cArray)
   {
      for (Color c : cArray)
      {
         this.colorList.add(c);
      }
   }


   /**
      Set each {@link Color} in this {@code Model}'s color list
      to the same {@link Color}.

      @param c  {@link Color} for all of this model's {@link Vertex} objects
   */
   public void setColor(Color c)
   {
      if (this.colorList.isEmpty())
      {
         for (int i = 0; i < this.vertexList.size(); i++)
         {
            this.colorList.add(c);
         }
      }
      else
      {
         for (int i = 0; i < this.colorList.size(); i++)
         {
            this.colorList.set(i, c);
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setColor(c);
      }
   }


   /**
      Set each {@link Color} in this {@code Model}'s color list
      to the same random {@link Color}.
   */
   public void setRandomColor()
   {
      Random generator = new Random();
      float r = generator.nextFloat();
      float g = generator.nextFloat();
      float b = generator.nextFloat();
      Color c = new Color(r, g, b);
      setColor(c);
   }


   /**
      Set each {@link Color} in this {@code Model}'s color list
      to a different random {@link Color}.
   */
   public void setRandomColors()
   {
      if (this.colorList.isEmpty())
      {
         setRandomVertexColors();
      }
      else
      {
         Random generator = new Random();
         for (int i = 0; i < this.colorList.size(); i++)
         {
            float r = generator.nextFloat();
            float g = generator.nextFloat();
            float b = generator.nextFloat();
            Color c = new Color(r, g, b);
            this.colorList.set(i, c);
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setRandomColors();
      }
   }


   /**
      Give each of this {@code Model}'s nested models
      a different random {@link Color}.
   */
   public void setRandomNestedModelColors()
   {
      Random generator = new Random();
      float r = generator.nextFloat();
      float g = generator.nextFloat();
      float b = generator.nextFloat();
      Color c = new Color(r, g, b);
      if (this.colorList.isEmpty())
      {
         for (int i = 0; i < this.vertexList.size(); i++)
         {
            this.colorList.add(c);
         }
      }
      else
      {
         for (int i = 0; i < this.colorList.size(); i++)
         {
            this.colorList.set(i, c);
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setRandomNestedModelColors();
      }
   }


   /**
      Set each {@link Vertex} in this {@code Model}
      to a different random {@link Color}.
      <p>
      NOTE: This will destroy whatever "color structure"
      the model might possess.
   */
   public void setRandomVertexColors()
   {
      this.colorList = new ArrayList<Color>();
      Random generator = new Random();
      for (int i = 0; i < this.vertexList.size(); i++)
      {
         float r = generator.nextFloat();
         float g = generator.nextFloat();
         float b = generator.nextFloat();
         Color c = new Color(r, g, b);
         this.colorList.add(c);
      }
      for (Primitive p : this.primitiveList)
      {
         for (int i = 0; i < p.vIndexList.size(); i++)
         {
            p.cIndexList.set(i, p.vIndexList.get(i));
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setRandomVertexColors();
      }
   }


   /**
      Set each {@link Primitive} in this {@code Model}
      to a different (uniform) random {@link Color}.
      <p>
      NOTE: This will destroy whatever "color structure"
      the model might possess.
   */
   public void setRandomPrimitiveColors()
   {
      this.colorList = new ArrayList<>();
      Random generator = new Random();
      for (Primitive p : this.primitiveList)
      {
         float r = generator.nextFloat();
         float g = generator.nextFloat();
         float b = generator.nextFloat();
         int index = colorList.size();
         this.colorList.add( new Color(r, g, b) );
         for (int i = 0; i < p.cIndexList.size(); i++)
         {
            p.cIndexList.set(i, index);
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setRandomPrimitiveColors();
      }
   }


   /**
      Set each {@link Primitive} in this {@code Model}
      to a different random {@link Color} at each vertex.
      <p>
      NOTE: This will destroy whatever "color structure"
      the model might possess.
   */
   public void setRainbowPrimitiveColors()
   {
      colorList = new ArrayList<>();
      Random generator = new Random();
      for (Primitive p : this.primitiveList)
      {
         p.cIndexList = new ArrayList<>();
         int cIndex = this.colorList.size();
         for(int i = 0; i < p.vIndexList.size(); i++)
         {
            float r = generator.nextFloat();
            float g = generator.nextFloat();
            float b = generator.nextFloat();
            Color c = new Color(r, g, b);
            this.colorList.add(c);
            p.cIndexList.add(cIndex++);
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setRainbowPrimitiveColors();
      }
   }


   /**
      Set the back face color of each orientable {@link Primitive} in
      this {@code Model} to the given {@link Color}. The given {@link Color}
      object {@code c} is added to this {@code Model}'s color list.

      @param c  back face {@link Color} for all of this model's orientable primitives
   */
   public void setBackFaceColor(Color c)
   {
      int bfColorIndex = colorList.size();
      colorList.add(c);
      for (Primitive p : primitiveList)
      {
         if (p instanceof OrientablePrimitive)
         {
            ((OrientablePrimitive)p).setBackFaceColorIndex(bfColorIndex);
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setBackFaceColor(c);
      }
   }


   /**
      Set the back face color of each orientable {@link Primitive} in
      this {@code Model} to the given {@link Color}s. The given {@link Color}
      objects are added to this {@code Model}'s color list.

      @param c0  back face {@link Color} for all of this model's orientable primitives
      @param c1  back face {@link Color} for all of this model's orientable primitives
      @param c2  back face {@link Color} for all of this model's orientable primitives
   */
   public void setBackFaceColor(Color c0, Color c1, Color c2)
   {
      int i = colorList.size();
      colorList.add(c0);
      colorList.add(c1);
      colorList.add(c2);
      for (Primitive p : primitiveList)
      {
         if (p instanceof OrientablePrimitive)
         {
            ((OrientablePrimitive)p).setBackFaceColorIndex(i, i+1, i+2);
         }
      }
      for (Model m : this.nestedModels)
      {
         m.setBackFaceColor(c0, c1, c2);
      }
   }


   /**
      Recursively set the value of {@code doBackFaceCulling} for
      this {@code Model} and all of its nested models.

      @param doBackFaceCulling  new boolean value for {@code doBackFaceCulling}
   */
   public void setBackFaceCulling(boolean doBackFaceCulling)
   {
      this.doBackFaceCulling = doBackFaceCulling;

      for (Model m : this.nestedModels)
      {
         m.setBackFaceCulling(doBackFaceCulling);
      }
   }


   /**
      Recursively set the value of {@code frontFacingIsCCW} for
      this {@code Model} and all of its nested models.

      @param frontFacingIsCCW  new boolean value for {@code frontFacingIsCCW}
   */
   public void setFrontFacingIsCCW(boolean frontFacingIsCCW)
   {
      this.frontFacingIsCCW = frontFacingIsCCW;

      for (Model m : this.nestedModels)
      {
         m.setFrontFacingIsCCW(frontFacingIsCCW);
      }
   }


   /**
      Recursively set the value of {@code facesHaveTwoSides} for
      this {@code Model} and all of its nested models.

      @param facesHaveTwoSides  new boolean value for {@code facesHaveTwoSides}
   */
   public void setFacesHaveTwoSides(boolean facesHaveTwoSides)
   {
      this.facesHaveTwoSides = facesHaveTwoSides;

      for (Model m : this.nestedModels)
      {
         m.setFacesHaveTwoSides(facesHaveTwoSides);
      }
   }


   /**
      Recursively convert this {@code Model} and all of its nested models
      into a {@link Point} cloud.

      @param radius  integer radius of the rasterized cloud points
   */
   public void toPointCloud(int radius)
   {
      for (int i = 0; i < primitiveList.size(); ++i)
      {
         Primitive p = primitiveList.get(i);
         Points pts = new Points();
         pts.radius = radius;
         // copy all the indices from p to pts
         for (int j = 0; j < p.vIndexList.size(); ++j)
         {
            pts.addIndices(p.vIndexList.get(j),
                           p.cIndexList.get(j));
         }
         primitiveList.set(i, pts);
      }

      for (Model m : this.nestedModels)
      {
         m.toPointCloud(radius);
      }
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code Model} object
   */
   @Override
   public String toString()
   {
      String result = "";
      result += "Model visibility is: " + visible + "\n";
      result += "doBackFaceCulling is set to: " + doBackFaceCulling + "\n";
      result += "frontFacingIsCCW is set to: " + frontFacingIsCCW + "\n";
      result += "facesHaveTwoSides is set to: " + facesHaveTwoSides + "\n";
      result += "Model has " + vertexList.size() + " vertices.\n";
      result += "Model has " + colorList.size() + " colors.\n";
      result += "Model has " + primitiveList.size() + " primitives.\n";
      result += "Model has " + nestedModels.size() + " nested models\n";
      int i = 0;
      for (Vertex v : this.vertexList)
      {
         result += i + ": " + v.toString();
         i++;
      }
      //result = "Printing out this Model's " + colortList.size() + " colors:\n";
      i = 0;
      for (Color c : this.colorList)
      {
         result += i + ": " + c.toString();
         i++;
      }
      //result = "Printing out this Model's " + primitiveList.size() + " Primitives:\n";
      for (Primitive p : this.primitiveList)
      {
         result += p.toString();
      }
      //result += "Printing out this Model's " + nestedModels.size() + " nested Models\n";
      for (Model m : this.nestedModels)
      {
         result += m.toString();
      }
      //result += "Done printing out Model\n";
      return result;
   }
}
