/*

*/

package renderer.scene;

/**
   This is the orthographic (parallel) projection matrix
   from our last renderer.
   <pre>{@code
     [ 1  0  0   0   ]
     [ 0  1  0   0   ]
     [ 0  0  1 1+near]
     [ 0  0  0   1   ]
   }</pre>
   It translates the near plane z = -near in camera coordinates to
   the clipping plane at z = +1 in NDC. In fact, the whole standard
   view parallelepiped in camera coordinates translates into the
   standard view parallelepiped in NDC coordinates (which are also
   clip in this case).
<p>
   In this renderer we want the near plane {@code z = -near} in camera
   coordinates to be transformed to the clipping plane at {@code z = +1}
   in NDC and, simultaneously, we want the far plane {@code z= -far} in
   camera coordinates to be transformed to the clipping plane at
   {@code z = -1} in NDC. In order to simultaneously transform both
   the near and far planes, we need both a translation factor and a
   scaling factor in the orthographic projection matrix,
   <pre>{@code
     [ 1  0  0  0 ]
     [ 0  1  0  0 ]
     [ 0  0  a  b ]
     [ 0  0  0  1 ]
   }</pre>
   Solve for {@code a} and {@code b} so that {@code z = -near} gets
   transformed (after perspective division) to {@code z = +1} and
   {@code z = -far} gets transformed (after perspective division) to
   {@code z = -1}. The matrix maps
   <pre>{@code
        (x, y, -near, 1) -> (x, y, -a*near + b, 1)
   and
        (x, y, -far, 1)  -> (x, y, -a*far  + b, 1)
   }</pre>
   and we want
   <pre>{@code
              -a*near + b = 1
   and
              -a*far + b = -1
   }</pre>
   Solve the first equation for {@code b}
   <pre>{@code
               b = 1 + a*near.
   }</pre>
   and substitute for {@code b} in the second equation
   <pre>{@code
         -a*far + 1 + a*near = -1
             -a*far + a*near = -2
             -a*(far - near) = -2
              a = 2 / (far - near).
   }</pre>
   And then
   <pre>{@code
               b = 1 + (2 * near) / (far - near)
                 = ((far - near) / (far - near)) + (2 * near) / (far - near)
                 = ((far - near) + (2 * near)) / (far - near)
                 = (far + near) / (far - near)
   }</pre>
*/
public class OrthographicProjectionMatrix
{
   /**
      This is a static factory method.
      <p>
      Construct this orthographic projection matrix.
      <pre>{@code
        [ 1  0  0  0 ]
        [ 0  1  0  0 ]
        [ 0  0  a  b ]
        [ 0  0  0  1 ]
      }</pre>
      where
      <pre>{@code
        a = 2/(far - near)
        b = (far + near)/(far - near).
      }</pre>

      @param near  distance from the camera to the near plane
      @param far   distance from the camera to the far plane
   */
   public static Matrix build(double near, double far)
   {
      double a = 2/(far - near);
      double b = (far + near)/(far - near);

      return Matrix.build(
               new Vector(1.0,  0.0,  0.0,  0.0),
               new Vector(0.0,  1.0,  0.0,  0.0),
               new Vector(0.0,  0.0,    a,  0.0),
               new Vector(0.0,  0.0,    b,  1.0) );
   }
}
