/*

*/

package renderer.scene;

/**
   This was our first perspective projection matrix.
   <pre>{@code
     [ 1  0  0  0 ]
     [ 0  1  0  0 ]
     [ 0  0  0  1 ]
     [ 0  0 -1  0 ]
   }</pre>
   This matrix transforms the "horizontal" three-dimensional
   camera space {@code w = 1} into the "vertical" three-dimensional
   homogeneous space {@code z = 1} (where here "horizontal" means
   perpendicular to the w-axis and "vertical" means parallel to the
   w-axis). Then perspective division transforms the "vertical" space
   {@code z = 1} back into the "horizontal" space {@code w = 1} in such
   a way that the standard infinite view pyramid from camera space is
   transformed into the standard infinite view parallelipiped used by
   the orthographic projection (in normalized-device-coordinates).
<p>
   This was the perspective projection matrix used in our last renderer.
   <pre>{@code
     [ 1  0  0   0   ]
     [ 0  1  0   0   ]
     [ 0  0  0  near ]
     [ 0  0 -1   0   ]
   }</pre>
   This matrix transforms the "horizontal" three-dimensional camera
   space {@code w = 1} into the "vertical" three-dimensional homogeneous
   space {@code z = near}. This also transforms the near plane
   {@code (x, y, -near, 1)} in camera coordinates to the plane
   {@code (x, y, near, near)} in clip coordinates, which satisfies the
   homogeneous equation {@code w - z = 0}. In the last renderer, this
   became the equation for the front clipping plane.
<p>
   For this renderer we want a projection matrix that transforms the near
   plane {@code z = -near} in camera coordinates, to the clipping plane
   {@code w - z = 0} in clip coordinates (the plane {@code z = +1} in NDC)
   and also tranforms the far plane {@code z = -far} in camera coordinates
   to the clipping plane {@code w + z = 0} in clip coordinates (the plane
   {@code z = -1} in NDC).
<p>
   Consider a projection matrix of the following form.
   <pre>{@code
     [ 1  0  0  0 ]
     [ 0  1  0  0 ]
     [ 0  0  a  b ]
     [ 0  0 -1  0 ]
   }</pre>
   This matrix can be factored as the product of the following two matrices.
   <pre>{@code
       [ 1  0  0  0 ]   [ 1  0  0  0 ]
     = [ 0  1  0  0 ] * [ 0  1  0  0 ]
       [ 0  0  1 -a ]   [ 0  0  0  b ]
       [ 0  0  0  1 ]   [ 0  0 -1  0 ]
   }</pre>
   The matrix on the right is a projection matrix that transforms the
   "horizontal" camera space {@code w = 1} into the "vertical" homogeneous
   space {@code z = b}. The matrix on the left is a skew matrix that skews
   the four-dimensional homogeneous space along the w-axis in the z-direction.
   In particular, it skews the "vertical" homogeneous space {@code z = b} so
   that it is no longer "vertical" and we want to skew it enough so that it
   intersects both the clipping planes {@code w - z = 0} and {@code w + z = 0}
   in the upper half of homogeneous space with {@code w > 0}.
<p>
   We need to choose the constants {@code a} and {@code b} so that the product
   matrix transforms the near plane {@code z = -near} in camera coordinates
   to the clipping plane {@code w - z = 0} (the plane {@code z = +1} in NDC)
   and also transforms the far plane {@code z = -far} in camera coordinates
   to the clipping plane {@code w + z = 0} (the plane {@code z = -1} in NDC).
<p>
   With the above product matrix the camera's near plane, {@code z = -near},
   gets transformed as
   <pre>{@code
        (x, y, -near, 1) -> (x, y, -a*near+b, near)
   }</pre>
   which, after prespective division is the plane
   <pre>{@code
        (x/near, y/near, -a+b/near, 1)
   }</pre>
   Also, with the above product matrix the camera's far plane,
   {@code z = -far}, gets transformed as
   <pre>{@code
        (x, y, -far, 1) -> (x, y, -a*far+b, far)
   }</pre>
   which, after prespective division is the plane
   <pre>{@code
        (x/far, y/far, -a+b/far, 1)
   }</pre>
   We need
   <pre>{@code
      -a + b/near =  1  and
      -a + b/far  = -1.
   }</pre>
   Solving the first equation for {@code a} we get
   <pre>{@code
      a = b/near - 1
   }</pre>
   Substituting for {@code a} in the second equation we get
   <pre>{@code
      -(b/near - 1) + b/far = -1
      -b/near + 1 + b/far = -1
      b(1/near - 1/far) = 2
      b(far - near)/(near*far) = 2
      b = 2*near*far/(far - near).
   }</pre>
   And then
   <pre>{@code
     a = (2*near*far/(far - near))/near - 1
       = (2*far/(far - near)) - 1
       = (2*far/(far - near)) - (far - near)/(far - near)
       = (2*far - (far - near))/(far - near)
       = (far + near) / (far - near).
   }</pre>
   So the final projection matrix looks like this (using {@code n}
   and {@code f} to abbreviate {@code near} and {@code far}).
   <pre>{@code
   [ 1  0      0          0      ]
   [ 0  1      0          0      ]
   [ 0  0 (f+n)/(f-n)  2nf/(f-n) ]
   [ 0  0     -1          0      ]
   }</pre>
<p>
   Here is a way to verify that the above matrix maps the camera's near
   and far planes to {@code z = 1} and {@code z = -1} in NDC. Start with
   a point {@code (x, y, z, 1)} in camera coordinates and apply the matrix
   transformation followed by perspective division.
   <pre>{@code
        (x, y, z, 1) -> ( x,    y,    z*(f+n)/(f-n) + 2nf/(f-n),  -z)
                     -> (-x/z, -y/z, -(f+n)/(f-n) - 2nf/((f-n)*z), 1)
   }</pre>
   Notice that the camera's near plane, {@code z = -near}, will be
   transformed to
   <pre>{@code
       z = -(f+n)/(f-n) + 2nf/((f-n)*n)
         = -(f+n)/(f-n) + 2f/(f-n)
         = (-f+2f-n)/(f-n)
         = (f-n)/(f-n)
         = 1.
   }</pre>
   Similarly, the camera's far plane, {@code z = -far}, will be
   transformed to
   <pre>{@code
       z = -(f+n)/(f-n) + 2nf/((f-n)*f)
         = -(f+n)/(f-n) + 2n/(f-n)
         = (-f-n+2n)/(f-n)
         = (-f+n)/(f-n)
         = -1.
   }</pre>
<p>
   Consider a projective ray from the origin in camera space through
   a point {@code (x, y, -1, 1)} in the camera's view plane.
   <pre>{@code
       (x*t, y*t, -t, 1)  with  0 < t < infinity.
   }</pre>
   Here is what this ray is transformed into by the matrix and perspective
   division.
   <pre>{@code
      (x*t, y*t, -t, 1)                                   (camera coordinates)
           -> (x*t, y*t, -t*(f+n)/(f-n) + 2nf/(f-n), t)   (clip coordinates)
           -> (x, y, -(f+n)/(f-n) + 2nf/((f-n)*t), 1)     (NDC)
   }</pre>
   When {@code t=near} this ray intersects the camera's near plane at the
   point {@code (x*near, y*near, -near, 1)} in camera coordinates and the
   transformed ray intersects the plane {@code z = 1} in NDC.
   When {@code t < near}, the point on the ray with camera coordinates
   {@code (x*t, y*t, -t, 1)} will be in front of the camera's near plane
   and the transformed point from the ray will satisfy the inequality
   {@code z > 1} in NDC. Similarly, when {@code t > far}, the point on the
   ray with camera coordinates {@code (x*t, y*t, -t, 1)} will be behind the
   camera's far plane, and the transformed point will satisfy the inequality
   {@code -1 > z} in NDC. This verifies that in NDC {@code z = 1} is the
   front clipping plane and {@code z = -1} is the back clipping plane.
<p>
   In the rendering pipeline, the projection transformation is preceded by the
   normalization transformation. Here is the product of these two matrices.
   <pre>{@code
           projection                        normalization

   [ 1  0      0          0      ]   [ 2n/(r-l)   0       (r+l)/(r-l)  0 ]
   [ 0  1      0          0      ] * [  0       2n/(t-b)  (t+b)/(t-b)  0 ]
   [ 0  0 (f+n)/(f-n)  2nf/(f-n) ]   [  0         0            1       0 ]
   [ 0  0     -1          0      ]   [  0         0            0       1 ]


        [ 2n/(r-l)    0       (r+l)/(r-l)     0      ]
      = [  0        2n/(t-b)  (t+b)/(t-b)     0      ]
        [  0          0       (f+n)/(f-n)  2nf/(f-n) ]
        [  0          0           -1          0      ]
   }</pre>
   This matrix is almost exactly the projection matrix used by OpenGL. The
   only difference is two minus signs in the third rwo of the OpenGL version
   of the projection matrix. The reson for this difference is that OpenGL
   uses a left handed clip coordinate system (so their projection matrix has
   a negative determinant and is orientation reversing).

   <a href="http://www.songho.ca/opengl/gl_projectionmatrix.html" target="_top">
            http://www.songho.ca/opengl/gl_projectionmatrix.html</a>

   <a href="https://www.scratchapixel.com/lessons/3d-basic-rendering/perspective-and-orthographic-projection-matrix/opengl-perspective-projection-matrix" target="_top">
            https://www.scratchapixel.com/lessons/3d-basic-rendering/perspective-and-orthographic-projection-matrix/opengl-perspective-projection-matrix</a>

   <a href="https://msdn.microsoft.com/en-us/library/dd373537(v=vs.85).aspx" target="_top">
            https://msdn.microsoft.com/en-us/library/dd373537(v=vs.85).aspx</a>

   <a href="https://stackoverflow.com/questions/4124041/is-opengl-coordinate-system-left-handed-or-right-handed" target="_top">
            https://stackoverflow.com/questions/4124041/is-opengl-coordinate-system-left-handed-or-right-handed</a>

   <a href="http://www.realtimerendering.com/blog/left-handed-vs-right-handed-viewing/" target="_top">
            http://www.realtimerendering.com/blog/left-handed-vs-right-handed-viewing/</a>
*/
public class PerspectiveProjectionMatrix
{
   /**
      This is a static factory method.
      <p>
      Construct this perspective projection matrix.
      <pre>{@code
        [ 1  0  0  0 ]
        [ 0  1  0  0 ]
        [ 0  0  a  b ]
        [ 0  0 -1  0 ]
      }</pre>
      where
      <pre>{@code
        a = (far+near)/(far-near)
        b = 2*near*far/(far-near)
      }</pre>

      @param near  distance from the camera to the near plane
      @param far   distance from the camera to the far plane
   */
   public static Matrix build(double near, double far)
   {
      double a = (far + near)/(far - near);
      double b =   2*near*far/(far - near);

      return Matrix.build(
              new Vector(1.0,  0.0,  0.0,  0.0),
              new Vector(0.0,  1.0,  0.0,  0.0),
              new Vector(0.0,  0.0,    a, -1.0),
              new Vector(0.0,  0.0,    b,  0.0) );
   }
}
