/*

*/

package renderer.scene;

import java.util.List;
import java.util.ArrayList;

/**
   A {@code Position} data structure represents a group of geometric objects
   positioned (both location and orientation) in three-dimensional space as
   part of a {@link Scene}. A {@code Position} is a recursive data structure.
   Every {@code Position} object contains a {@link List} of nested
   {@code Position} objects. The list of nested {@code Position}s lets us
   define what are called hierarchical, or nested, scenes. These are scenes
   that are made up of groups of models and each group can be moved around
   in the scene as a single unit while the individual models in the group
   can also be moved around within the group.
<p>
   A {@code Position} object holds references to a {@link Model} object,
   a {@link Matrix} object, and a {@code List} of {@code Position} objects.
   A {@code Position}'s {@code List} of {@code Position} objects creates
   a tree data structure of {@code Position} objects. A {@code Position}'s
   {@link Model} object represents a geometric shape in the {@link Scene}.
   The role of a {@code Position}'s  {@link Matrix} can be understood two
   ways. First, the {@link Matrix} determines the {@link Model}'s location
   and orientation within the local coordinate system determined by the
   {@code Position}'s parent {@code Position} (in the {@link Scene}'s forest
   of {@code Position} objects). Second, the {@link Matrix} determines a new
   local coordinate system within which the {@link Model} (and all the nested
   models lower in the tree) is plotted. The two ways of understanding a
   {@code Position}'s  {@link Matrix} correspond to reading a matrix
   transformation expression
   <pre>{@code
                T * v
   }</pre>
   from either right-to-left or left-to-right.
<p>
   When the renderer renders a {@code Position} object, the renderer
   traverses the tree of {@code Position}s rooted at the given
   {@code Position}. The renderer does a recursive, pre-order
   depth-first-traversal of the tree. As the renderer traverses the tree,
   it accumulates a "current-transformation-matrix" that multiplies each
   {@code Position}'s {@link Matrix} along the path from the tree's root
   {@code Position} to wherever the traversal is in the tree (this is the
   "pre-order" step in the traversal). The {@code ctm} is the current
   model-to-view transformation {@link Matrix}. The first stage of the
   rendering pipeline, {@link renderer.pipeline.Model2View}, multiplies every
   {@link Vertex} in a {@link Model}'s vertex list by this {@code ctm}, which
   converts the coordinates in each {@link Vertex} from the model's own local
   coordinate system to the {@code Camera}'s view coordinate system (which is "shared"
   by all the other models). Multiplication by the {@code ctm} has the effect
   of "placing" the model in view space at an appropriate location (using the
   translation part of the {@code ctm}) and in the appropriate orientation
   (using the rotation part of the {@code ctm}). Notice the difference between
   a {@code Position}'s {@code ctm} and the {@code Position}'s {@link Matrix}.
   At any specific node in the {@link Scene}'s forest of {@code Position} nodes,
   the {@code Position}'s {@link Matrix} places the {@code Position}'s
   {@link Model} within the local coordinate system of the {@code Position}'s
   parent {@code Position}. But the {@code Position}'s {@code ctm} places the
   {@code Position}'s {@link Model} within the {@link Camera}'s view coordinate
   system.
*/
public class Position
{
   public Model  model;
   public Matrix modelMatrix;
   public List<Position> nestedPositions;

   public boolean visible;

   /**
      Construct a default {@code Position} with the identity {@link Matrix},
      no {@link Model} object, and no nested {@code Position}s.
   */
   public Position()
   {
      this.model = null;
      this.modelMatrix = Matrix.identity(); // identity matrix
      this.nestedPositions = new ArrayList<>();
      this.visible = true;
   }


   /**
      Construct a {@code Position} with the identity {@link Matrix},
      the given {@link Model} object, and no nested {@code Position}s.

      @param model  {@link Model} object to place at this {@code Position}
   */
   public Position(Model model)
   {
      this.model = model;
      this.modelMatrix = Matrix.identity(); // identity matrix
      this.nestedPositions = new ArrayList<>();
      this.visible = true;
   }


   /**
      A "copy constructor". This constructor should make a deep copy
      of the given {@code Position}'s {@link Matrix}, {@link Model}
      and {@link List} of nested {@code Position}s.

      @param position  {@code Position} to make a copy of
   */
   public Position(Position position) // a "copy constructor"
   {
      if (null != position.model)
      {
         this.model = new Model(position.model); // deep copy of the Model
      }
      else
      {
         this.model = null;
      }
      this.modelMatrix = new Matrix(position.modelMatrix);
      this.nestedPositions = new ArrayList<>();
      for (Position p : position.nestedPositions)
      {
         this.nestedPositions.add( new Position(p) ); // deep copy of each Position
      }
      this.visible = position.visible;
   }


   /**
      Add a nested {@code Position} (or Positions) to this {@code Position}'s
      {@link List} of nested {@code Position}s.

      @param pArray  array of nested {@code Position}s to add to this {@code Position}
   */
   public void addNestedPosition(Position... pArray)
   {
      for (Position p : pArray)
      {
         this.nestedPositions.add(p);
      }
   }


   /**
      Get the index to a nested {@code Position} from this {@code Position}'s
      {@link List} of nested {@code Position}s.

      @param p  {@code Position} to search for in this {@code Position}'s {@link List} of nested {@code Position}s
      @return the index of the first occurrence of the specified {@code Position},
              or -1 if the {@code Position} list does not contain the {@code Position}
   */
   public int indexOfNestedPosition(Position p)
   {
      return this.nestedPositions.indexOf(p);
   }


   /**
      Get a reference to the nested {@code Position} at the given index in this {@code Position}'s
      {@link List} of nested {@code Position}s.

      @param index  index of the nested {@code Position} to return
      @return nested {@code Position} at the specified index in the {@link List} of nested {@code Position}s
      @throws IndexOutOfBoundsException if the index is out of range
              {@code (index < 0 || index >= size())}
   */
   public Position getNestedPosition(int index)
   {
      return nestedPositions.get(index);
   }


   /**
      Replace the nested {@code Position} at the given index in this {@code Position}'s
      {@link List} of nested {@code Position}s.

      @param index  index of the nested {@code Position} to replace
      @param p      {@code Position} to be stored at the specified index
      @return {@code Position} previously at the specified index
      @throws IndexOutOfBoundsException if the index is out of range
              {@code (index < 0 || index >= size())}
   */
   public Position setNestedPosition(int index, Position p)
   {
      return this.nestedPositions.set(index, p);
   }


   /**
      Reset this {@code Position}'s {@link Matrix} to the identity matrix.

      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position matrix2Identity()
   {
      this.modelMatrix = Matrix.identity();
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by the
      parameter {@code m} and make the result the new {@link Matrix}.

      @param m  {@link Matrix} to multiply this {@code Position}'s model matrix by
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position matrixMult(Matrix m)
   {
      this.modelMatrix = this.modelMatrix.times(m);
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by a rotation
      matrix and make the result the new {@link Matrix}.
      <p>
      The rotation is by angle {@code theta} around the x-axis.

      @param theta  angle (in degrees) to rotate this {@code Position} by
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position rotateX(double theta)
   {
      this.matrixMult( Matrix.rotateX(theta) );
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by a rotation
      matrix and make the result the new {@link Matrix}.
      <p>
      The rotation is by angle {@code theta} around the y-axis.

      @param theta  angle (in degrees) to rotate this {@code Position} by
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position rotateY(double theta)
   {
      this.matrixMult( Matrix.rotateY(theta) );
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by a rotation
      matrix and make the result the new {@link Matrix}.
      <p>
      The rotation is by angle {@code theta} around the z-axis.

      @param theta  angle (in degrees) to rotate this {@code Position} by
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position rotateZ(double theta)
   {
      this.matrixMult( Matrix.rotateZ(theta) );
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by a rotation
      matrix and make the result the new {@link Matrix}.
      <p>
      The rotation is by angle {@code theta} with axis of rotation
      given by the vector {@code (x, y, z)}.

      @param theta  angle (in degrees) to rotate this {@code Position} by
      @param x      x-coordinate of a vector on the axis of rotation
      @param y      y-coordinate of a vector on the axis of rotation
      @param z      z-coordinate of a vector on the axis of rotation
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position rotate(double theta, double x, double y, double z)
   {
      this.matrixMult( Matrix.rotate(theta, x, y, z) );
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by a scaling
      matrix and make the result the new {@link Matrix}.
      <p>
      The scaling matrix scales by the amounts {@code x}, {@code y} and
      {@code z} along the x-axis, y-axis, and z-axis (respectively).

      @param x  scaling factor for the x-axis
      @param y  scaling factor for the y-axis
      @param z  scaling factor for the z-axis
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position scale(double x, double y, double z)
   {
      this.matrixMult( Matrix.scale(x, y, z) );
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by a scaling
      matrix and make the result the new {@link Matrix}.
      <p>
      The scaling matrix scales by the amount {@code s} along each of
      the x-axis, y-axis, and z-axis.

      @param s  scaling factor for all three directions
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position scale(double s)
   {
      this.matrixMult( Matrix.scale(s) );
      return this;
   }

   /**
      Multiply this {@code Position}'s current {@link Matrix} by a translation
      matrix and make the result the new {@link Matrix}.
      <p>
      The translation matrix translates by the amounts {@code x}, {@code y},
      and {@code z} along the x-axis, y-axis, and z-axis (respectively).

      @param x  translation amount along the x-axis
      @param y  translation amount along the y-axis
      @param z  translation amount along the z-axis
      @return a reference to this {@code Position} to facilitate chaining method calls
   */
   public Position translate(double x, double y, double z)
   {
      this.matrixMult( Matrix.translate(x, y, z) );
      return this;
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code Position} object
   */
   @Override
   public String toString()
   {
      String result = "";
      result += "This Position's visibility is: " + visible + "\n";
      result += "This Position's model Matrix is\n";
      result += modelMatrix;
      result += "This Position's Model is\n";
      result += (null == model) ? "null\n" : model;
      result += "This Position has " + nestedPositions.size() + " nested Positions\n";
      for (Position p : this.nestedPositions)
      {
         result += p.toString();
      }
      return result;
   }
}
