/*

*/

package renderer.scene;

/**
   A {@code Vertex} object has four doubles which represent the
   homogeneous coordinates of a point in 3-dimensional space.
   The fourth, homogeneous, coordinate will usually be 1, but in
   some stages of the graphics rendering pipeline it can be some
   other (non-zero) number.
<p>
   When a {@code Vertex} object is created in a client program,
   before the {@code Vertex} object moves down the graphics rendering
   pipeline, the coordinates in the {@code Vertex} will be in some
   model's local coordinate system.
<p>
   As a {@code Vertex} object moves down the graphics rendering
   pipeline, the coordinates in the {@code Vertex} will be transformed
   from one coordinate system to another.
*/
public class Vertex
{
   public double x, y, z, w;   // a vertex in homogenous coordinates

   /**
      Construct a default {@code Vertex}.
   */
   public Vertex()
   {
      set(0.0, 0.0, 0.0, 1.0);
   }


   /**
      Construct a new {@code Vertex} (with homogeneous coordinates)
      using the given {@code x}, {@code y}, and {@code z} coordinates.

      @param x  x-coordinate of the new {@code Vertex}
      @param y  y-coordinate of the new {@code Vertex}
      @param z  z-coordinate of the new {@code Vertex}
   */
   public Vertex(double x, double y, double z)
   {
      set(x, y, z, 1.0);
   }


   /**
      Construct a new {@code Vertex} with the given homogeneous coordinates.

      @param x  x-coordinate of the new {@code Vertex}
      @param y  y-coordinate of the new {@code Vertex}
      @param z  z-coordinate of the new {@code Vertex}
      @param w  w-coordinate of the new {@code Vertex}
   */
   public Vertex(double x, double y, double z, double w)
   {
      set(x, y, z, w);
   }


   /**
      Construct a new {@code Vertex} that is a copy of another {@code Vertex}.

      @param v  {@code Vertex} to make a copy of
   */
   public Vertex(Vertex v) // a "copy constructor"
   {
      set(v.x, v.y, v.z, v.w);
   }


   /**
      Set the {@code x}, {@code y}, and {@code z} coordinates
      of this {@code Vertex}.

      @param x  new x-coordinate for this {@code Vertex}
      @param y  new y-coordinate for this {@code Vertex}
      @param z  new z-coordinate for this {@code Vertex}
   */
   public void set(double x, double y, double z)
   {
      set(x, y, z, 1);
   }


   /**
      Set the homogeneous coordinates of this {@code Vertex}.

      @param x  new x-coordinate for this {@code Vertex}
      @param y  new y-coordinate for this {@code Vertex}
      @param z  new z-coordinate for this {@code Vertex}
      @param w  new w-coordinate for this {@code Vertex}
   */
   public void set(double x, double y, double z, double w)
   {
      this.x = x;
      this.y = y;
      this.z = z;
      this.w = w;
   }


   /**
      {@link Matrix} times {@code Vertex} mutates this {@code Vertex}.
      <p>
      Needed by the linear transformation stages of the graphics pipeline.

      @param matrix  {@code Matrix} to multiply this {@code Vertex} by
   */
   public void times(Matrix matrix)
   {
      Vector col1 = matrix.v1;
      Vector col2 = matrix.v2;
      Vector col3 = matrix.v3;
      Vector col4 = matrix.v4;

      // dot product of the first matrix row with this vertex
      double x2 = (col1.x * this.x) + (col2.x * this.y)  + (col3.x * this.z) + (col4.x * this.w);

      // dot product of the second matrix row with this vertex
      double y2 = (col1.y * this.x) + (col2.y * this.y)  + (col3.y * this.z) + (col4.y * this.w);

      // dot product of the third matrix row with this vertex
      double z2 = (col1.z * this.x) + (col2.z * this.y)  + (col3.z * this.z) + (col4.z * this.w);

      // dot product of the fourth matrix row with this vertex
      double w2 = (col1.w * this.x) + (col2.w * this.y)  + (col3.w * this.z) + (col4.w * this.w);

      // Now mutate this Vertex object.
      this.x = x2;
      this.y = y2;
      this.z = z2;
      this.w = w2;
   }


   /**
      Mutate this {@code Vertex} by doing a perspective division of
      its homogeneous coordinates.
      <p>
      Needed by the {@link renderer.pipeline.PerspectiveDivision} stage in
      the rendering pipeline.
   */
   public void perspectiveDivision()
   {
      double r = 1.0 / w;  // the reciprocal of the w-coordinate
      this.x *= r;
      this.y *= r;
      this.z *= r;
      this.w  = 1.0;
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code Vertex} object
   */
   @Override
   public String toString()
   {
      int precision = 5;  // the default precision for the format string
      return toString(precision);
   }


   /**
      For debugging.
      <p>
      Allow the precision of the formatted output to be specified.

      @param precision  precision value for the format string
      @return {@link String} representation of this {@code Vertex} object
   */
   public String toString(int precision)
   {
      int iWidth = 3; // default width of integer part of the format string
      return toString(precision, iWidth);
   }


   /**
      For debugging.
      <p>
      Allow the precision and width of the formatted output to be specified.
      By width, we mean the width of the integer part of each number.

      @param precision  precision value for the format string
      @param iWidth     width of the integer part of the format string
      @return {@link String} representation of this {@code Vertex} object
   */
   public String toString(int precision, int iWidth)
   {
      // Here is one way to get programmable precision and width.
      int p = precision;      // the precision for the following format string
      int t = p + iWidth + 2; // the width for the following format string
      String format = "(x,y,z,w)=(% "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f)\n";
      return String.format(format, x, y, z, w);
   }
}
