/*

*/

package renderer.scene.primitives;
import  renderer.scene.Model;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;

/**
   A {@code LineSegment} object has four integers that
   represent the endpoints of the line segment and the
   color at each endpoint. Two of the integers are indices
   into the {@link Vertex} list of a {@link Model} object
   and the other two integers are indices into the {@link Color}
   list of that {@link Model} object.
*/
public class LineSegment extends Primitive
{
   /**
      Construct a {@code LineSegment} object using two integer indices.
      Use the given indices for both the vertex and the color lists.

      @param i0  index of 1st endpoint vertex and color of the new {@code LineSegment}
      @param i1  index of 2nd endpoint vertex and color of the new {@code LineSegment}
   */
   public LineSegment(int i0, int i1)
   {
      this(i0, i1, i0, i1);
   }


   /**
      Construct a {@code LineSegment} object using two integer indices
      for the vertices and one integer index for the colors.

      @param v0  index of 1st endpoint vertex of the new {@code LineSegment}
      @param v1  index of 2nd endpoint vertex of the new {@code LineSegment}
      @param c0  index of the color of the new {@code LineSegment}
   */
   public LineSegment(int v0, int v1, int c)
   {
      this(v0, v1, c, c);
   }


   /**
      Construct a {@code LineSegment} object using two integer indices
      for the vertices and two integer indices for the colors.
      <p>
      NOTE: This method does not put any Vertex or Color objects into
      this LineSegment's Model. This method assumes that the given indices
      are valid (or will be valid by the time this LineSegment gets rendered).

      @param v0  index of 1st endpoint vertex of the new {@code LineSegment}
      @param v1  index of 2nd endpoint vertex of the new {@code LineSegment}
      @param c0  index of 1st endpoint color of the new {@code LineSegment}
      @param c1  index of 2nd endpoint color of the new {@code LineSegment}
   */
   public LineSegment(int v0, int v1, int c0, int c1)
   {
      super();

      vIndexList.add(v0);
      vIndexList.add(v1);
      cIndexList.add(c0);
      cIndexList.add(c1);
   }


   /**
      Implement the {@link #makeCopy()} method from the {@link Primitive} abstract base class.
      <p>
      Construct a deep copy of this {@code LineSgement} object.

      @return a {@code LineSegment} object that is a deep copy of this {@code LineSegment} object
   */
   @Override
   public LineSegment makeCopy() // a kind of "copy constructor"
   {
      return new LineSegment(this.vIndexList.get(0),
                             this.vIndexList.get(1),
                             this.cIndexList.get(0),
                             this.cIndexList.get(1));
   }


   /**
      Give this {@code LineSegment} the given {@code Color}s at its two endpoints.
      <p>
      This method adds the given color objects to the model's color list.
      <p>
      Warning: Repeatedly calling this method will leave the Model's color
      list bloated with unused Color objects. To repeatedly change this line
      segment's colors, use {@link #changeColors()}.

      @param model   {@link Model} object that this {@code LineSegment} is part of
      @param color0  new {@link Color} object for 1st endpoint of this {@code LineSegment}
      @param color1  new {@link Color} object for 2nd endpoint of this {@code LineSegment}
   */
   public void addColors(Model model, Color color0, Color color1)
   {
      int index = model.colorList.size();
      model.colorList.add(color0);
      model.colorList.add(color1);
      this.cIndexList.set(0, index);
      this.cIndexList.set(1, index + 1);
   }


   /**
      Give this {@code LineSegment} the given {@code Color}s at its two endpoints.
      <p>
      Warning: This method replaces the color objects this llne segment
      is using in the model's color list. So calling this method will
      also change the color of any other {@link Primitive} that shares
      a color object with this line segment. To safely change only this
      line segment's colors, use {@link #addColors()}.

      @param model   {@link Model} object that this {@code LineSegment} is part of
      @param color0  new {@link Color} object for 1st endpoint of this {@code LineSegment}
      @param color1  new {@link Color} object for 2nd endpoint of this {@code LineSegment}
   */
   public void changeColors(Model model, Color color0, Color color1)
   {
      model.colorList.set(this.cIndexList.get(0), color0);
      model.colorList.set(this.cIndexList.get(1), color1);
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code LineSegment} object
   */
   @Override
   public String toString()
   {
      return "LineSegment: ([" + vIndexList.get(0) + ", " + vIndexList.get(1) + "], "
                         + "[" + cIndexList.get(0) + ", " + cIndexList.get(1) + "])\n";
   }
}
