/*

*/

package renderer.scene.primitives;
import  renderer.scene.Model;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;
import java.util.Random;

/**
   A {@code OrientablePrimitive} is a {@link Primitive} that can
   be given an orientation, so it has a both a front and a back side.
<p>
   We have five orientable geometric primitives, {@link Face},
   {@link Triangle}, {@link TriangleStrip}, {@link TriangleFan},
   and {@link Triangles}.
<p>
   Each {@code OrientablePrimitive} holds three lists of integer indices.
<p>
   One list is of indices into its {@link Model}'s {@link List}
   of {@link Vertex} objects. These are the vertices that
   determine the primitive's geometry.
<p>
   The other two lists are lists of indices into its {@link Model}'s
   {@link List} of {@link Color} objects. One index list determines
   the primitive's front facing colors and the other index list determines
   the primitive's back facing colors.
<p>
   The three lists of integer indices must always have the same length.
   For every {@link Vertex} index in this {@code OrientablePrimitive}
   there must be a front {@link Color} index and a back {@link Color}
   index.
*/
public abstract class OrientablePrimitive extends Primitive
{
   // A OrientablePrimitive object has an extra color list for back facing colors.
   public List<Integer> cIndexList2; // indices for this primitive's back facing colors

   /**
      Construct an empty {@code OrientablePrimitive}.
   */
   protected OrientablePrimitive()
   {
      this.cIndexList2 = new ArrayList<>();
   }


   /**
      Construct a {@code OrientablePrimitive} with the given array of
      indices for the vertex and color index lists.
      <p>
      NOTE: This constructor does not put any Vertex or Color objects into
      this primitive's Model. This constructor assumes that the given indices
      are valid (or will be valid by the time this primitive gets rendered).

      @param indices  array of vertex and color indices to place in this {@code OrientablePrimitive}
   */
   protected OrientablePrimitive(int... indices)
   {
      this();

      for (int i : indices)
      {
         addIndices(i, i, i);
      }
   }


   /**
      Add the given array of indices to the vertex and color index lists.
      <p>
      NOTE: This method does not put any Vertex or Color objects into
      this primitive's Model. This method assumes that the given indices
      are valid (or will be valid by the time this Face gets rendered).

      @param indices  array of vertex and color indices to add to this {@code OrientablePrimitive}
   */
   @Override
   public void addIndex(int... indices)
   {
      for (int i : indices)
      {
         addIndices(i, i, i);
      }
   }


   /**
      Add the given indices to the vertex and color index lists.
      <p>
      NOTE: This method does not put any Vertex or Color objects into
      this primitive's Model. This method assumes that the given indices
      are valid (or will be valid by the time this primitive gets rendered).

      @param vIndex  integer vertex index to add to this {@code OrientablePrimitive}
      @param cIndex  integer color index to add to this {@code OrientablePrimitive}
   */
   @Override
   public void addIndices(int vIndex, int cIndex)
   {
      addIndices(vIndex, cIndex, cIndex);
   }


   /**
      Add the given indices to the vertex and color index lists.
      <p>
      NOTE: This method does not put any Vertex or Color objects into
      this primitive's Model. This method assumes that the given indices
      are valid (or will be valid by the time this primitive gets rendered).

      @param vIndex  integer vertex index to add to this {@code OrientablePrimitive}
      @param cIndex  integer color index to add to this {@code OrientablePrimitive}
      @param bIndex  integer back face color index to add to this {@code OrientablePrimitive}
   */
   public void addIndices(int vIndex, int cIndex, int bIndex)
   {
      vIndexList.add(vIndex);
      cIndexList.add(cIndex);
      cIndexList2.add(bIndex);
   }


   /**
      Give this {@code OrientablePrimitive} the uniform back face color
      indexed by the given color index.
      <p>
      NOTE: This method does not put a Color object into this primitive's
      Model. This method assumes that the given index is valid (or will be
      valid by the time this primitive gets rendered).

      @param cIndex  integer index into a {@link Model}'s color list
   */
   public void setBackFaceColorIndex(int cIndex)
   {
      for (int i = 0; i < cIndexList2.size(); i++)
      {
         cIndexList2.set(i, cIndex);
      }
   }


   /**
      Give the triangles of this {@code OrientablePrimitive} the back face
      colors indexed by the given color indices.
      <p>
      NOTE: This method does not put a Color object into this primitive's
      Model. This method assumes that the given index is valid (or will be
      valid by the time this primitive gets rendered).

      @param c0  integer index into a {@link Model}'s color list
      @param c1  integer index into a {@link Model}'s color list
      @param c2  integer index into a {@link Model}'s color list
   */
   public void setBackFaceColorIndex(int c0, int c1, int c2)
   {
      int[] c = {c0, c1, c2};
      for (int i = 0; i < cIndexList2.size(); ++i)
      {
         cIndexList2.set(i, c[i % 3]);
      }
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code OrientablePrimitive} object
   */
   @Override
   public abstract String toString();
}
