/*

*/

package renderer.scene.primitives;
import  renderer.scene.Model;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;
import java.util.Random;

/**
   A {@code Primitive} is something that we can build
   geometric shapes out of (a "graphics primitive").
<p>
   We have seven gemetric primitives, {@link LineSegment},
   {@link LineStrip}, {@link LineLoop}, {@link LineFan},
   {@link Lines}, {@link Point} and {@link Points}.
<p>
   Each {@code Primitive} holds two lists of integer indices.
<p>
   One list is of indices into its {@link Model}'s {@link List}
   of {@link Vertex} objects. These are the vertices that
   determine the primitive's geometry.
<p>
   The other list is of indices into its {@link Model}'s
   {@link List} of {@link Color} objects.
<p>
   The two lists of integer indices must always have the same length.
   For every {@link Vertex} index in this {@code Primitive} there
   must be a {@link Color} index.
*/
/*
   NOTE: The Primitive class could be an inner class of
         the Model class. Then each Primitive object would
         automatically have acces to the actual Vertex list
         that the Primitive is indexing into.
*/
public abstract class Primitive
{
   // A Primitive object is made up of indices to vertices and colors in a Model.
   public List<Integer> vIndexList;  // indices for this primitive's vertices
   public List<Integer> cIndexList;  // indices for this primitive's colors

   /**
      Construct an empty {@code Primitive}.
   */
   protected Primitive()
   {
      this.vIndexList = new ArrayList<>();
      this.cIndexList = new ArrayList<>();
   }


   /**
      Construct a {@code Primitive} with the given array of indices for
      the vertex and color index lists.
      <p>
      NOTE: This constructor does not put any Vertex or Color objects into
      this Primitive's Model. This constructor assumes that the given indices
      are valid (or will be valid by the time this Primitive gets rendered).

      @param indices  array of vertex and color indices to place in this {@code Primitive}
   */
   protected Primitive(int... indices)
   {
      this();

      for (int i : indices)
      {
         addIndices(i, i);
      }
   }


   /**
      Construct a (polymorphic) deep copy of this {@code Primitive}
      for use in another {@link Model}.
      <p>
      NOTE: By making this an abstract method we can define this method to
      produce polymorphic deep copies. In other words, when we call makeCopy()
      on a LineStrip object, we want to get back another LineStrip object and
      not a Primitive object.
      See
      <br> <a href="https://www.artima.com/intv/bloch13.html" target="_top">
                    https://www.artima.com/intv/bloch13.html</a>

      @return an object that is a deep copy of this {@code Primitive} object
   */
   public abstract Primitive makeCopy(); // a kind of "copy constructor"


   /**
      Add the given array of indices to the vertex and color index lists.
      <p>
      NOTE: This method does not put any Vertex or Color objects into
      this Primitive's Model. This method assumes that the given indices
      are valid (or will be valid by the time this Primitive gets rendered).

      @param indices  array of vertex and color indices to add to this {@code Primitive}
   */
   public void addIndex(int... indices)
   {
      for (int i : indices)
      {
         addIndices(i, i);
      }
   }


   /**
      Add the given indices to the vertex and color index lists.
      <p>
      NOTE: This method does not put any Vertex or Color objects into
      this Primitive's Model. This method assumes that the given indices
      are valid (or will be valid by the time this Primitive gets rendered).

      @param vIndex  integer vertex index to add to this {@code Primitive}
      @param cIndex  integer color index to add to this {@code Primitive}
   */
   public void addIndices(int vIndex, int cIndex)
   {
      vIndexList.add(vIndex);
      cIndexList.add(cIndex);
   }


   /**
      Give this {@code Primitive} the uniform color indexed by the
      given color index.
      <p>
      NOTE: This method does not put a Color object into this Primitive's
      Model. This method assumes that the given index is valid (or will be
      valid by the time this Primitive gets rendered).
   */
   public void setColorIndex(int cIndex)
   {
      for (int i = 0; i < cIndexList.size(); i++)
      {
         cIndexList.set(i, cIndex);
      }
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code Primitive} object
   */
   @Override
   public abstract String toString();
}
