/*

*/

package renderer.scene.primitives;
import  renderer.scene.Model;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;

/**
   A {@code Triangle} object has two lists of three integers each.
   One list represents the three corners of the triangle in space.
   The integers are indices into the {@link Vertex} list of a
   {@link Model} object. The other list holds integer indices
   into the {@link Color} list of that {@link Model} object.
*/
public class Triangle extends OrientablePrimitive
{
   /**
      Construct a {@code Triangle} object using three integer indices.
      Use the given indices for both the vertex and the color lists.

      @param i0  index for 1st {@link Vertex} of the new {@code Triangle}
      @param i1  index for 2nd {@link Vertex} of the new {@code Triangle}
      @param i2  index for 3rd {@link Vertex} of the new {@code Triangle}
   */
   public Triangle(int i0, int i1, int i2)
   {
      this(i0, i1, i2, i0, i1, i2);
   }


   /**
      Construct a {@code Triangle} object using three integer indices
      for the vertices and one integer index for the colors.

      @param v0  index of 1st {@link Vertex} of the new {@code Triangle}
      @param v1  index of 2nd {@link Vertex} of the new {@code Triangle}
      @param v2  index of 3rd {@link Vertex} of the new {@code Triangle}
      @param c   index of the {@link Color} of the new {@code Triangle}
   */
   public Triangle(int v0, int v1, int v2, int c)
   {
      this(v0, v1, v2, c, c, c);
   }


   /**
      Construct a {@code Triangle} object using three integer indices
      for the vertices and three integer indices for the colors.
      <p>
      NOTE: This method does not put any Vertex or Color objects into
      this Triangle's Model. This method assumes that the given indices
      are valid (or will be valid by the time this Triangle gets rendered).

      @param v0  index of 1st {@link Vertex} of the new {@code Triangle}
      @param v1  index of 2nd {@link Vertex} of the new {@code Triangle}
      @param v2  index of 3rd {@link Vertex} of the new {@code Triangle}
      @param c0  index of 1st {@link Color} of the new {@code Triangle}
      @param c1  index of 2nd {@link Color} of the new {@code Triangle}
      @param c2  index of 3rd {@link Color} of the new {@code Triangle}
   */
   public Triangle(int v0, int v1, int v2, int c0, int c1, int c2)
   {
      vIndexList.add(v0);
      vIndexList.add(v1);
      vIndexList.add(v2);
      cIndexList.add(c0);
      cIndexList.add(c1);
      cIndexList.add(c2);
      cIndexList2.add(c0);
      cIndexList2.add(c1);
      cIndexList2.add(c2);
   }


   /**
      Implement the {@link #makeCopy()} method from the {@link Primitive} abstract base class.
      <p>
      Construct a deep copy of this {@code Triangle} object.

      @return a {@code Triangle} object that is a deep copy of this {@code Triangle} object
   */
   @Override
   public Triangle makeCopy() // a kind of "copy constructor"
   {
      Triangle result = new Triangle(this.vIndexList.get(0),
                                     this.vIndexList.get(1),
                                     this.vIndexList.get(2),
                                     this.cIndexList.get(0),
                                     this.cIndexList.get(1),
                                     this.cIndexList.get(2));

      result.setBackFaceColorIndex(this.cIndexList2.get(0),
                                   this.cIndexList2.get(1),
                                   this.cIndexList2.get(2));

      return result;
   }


   /**
      Give this {@code Triangle} the uniform back face color indexed by the
      given color index.
      <p>
      NOTE: This method does not put a Color object into this Triangle's
      Model. This method assumes that the given index is valid (or will be
      valid by the time this Triangle gets rendered).

      @param cIndex  integer index into a {@link Model}'s color list
   */
   @Override
   public void setBackFaceColorIndex(int cIndex)
   {
      setBackFaceColorIndex(cIndex, cIndex, cIndex);
   }


   /**
      Give this {@code Triangle} the back face colors indexed by the
      given color indices.
      <p>
      NOTE: This method does not put a Color object into this Triangle's
      Model. This method assumes that the given index is valid (or will be
      valid by the time this Triangle gets rendered).

      @param c0  integer index into a {@link Model}'s color list
      @param c1  integer index into a {@link Model}'s color list
      @param c2  integer index into a {@link Model}'s color list
   */
   @Override
   public void setBackFaceColorIndex(int c0, int c1, int c2)
   {
      cIndexList2.set(0, c0);
      cIndexList2.set(1, c1);
      cIndexList2.set(2, c2);
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code Triangle} object
   */
   @Override
   public String toString()
   {
      return "Triangle: ([" + vIndexList.get(0) + ", "
                            + vIndexList.get(1) + ", "
                            + vIndexList.get(2) + "], ["
                            + cIndexList.get(0) + ", "
                            + cIndexList.get(1) + ", "
                            + cIndexList.get(2) + "], ["
                            + cIndexList2.get(0) + ", "
                            + cIndexList2.get(1) + ", "
                            + cIndexList2.get(2) + "])\n";
   }
}
