/*

*/

package renderer.scene.primitives;
import  renderer.scene.Model;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;

/**
   A {@code TriangleFan} object has {@code n+2} integers that
   represent the vertices of {@code n} triangles that
   radiate from a central point and another {@code n+1}
   integers that represent the color at each of those vertices.
<p>
   The integers are indices into the {@link Vertex} and {@link Color}
   lists of a {@link Model} object.
<p>
   See
     <a href="https://en.wikipedia.org/wiki/Triangle_fan" target="_top">
              https://en.wikipedia.org/wiki/Triangle_fan</a>
*/
public class TriangleFan extends OrientablePrimitive
{
   /**
      Construct a {@code TriangleFan} with the given array of indices for
      the vertex and color index lists.
      <p>
      NOTE: This constructor does not put any Vertex or Color objects into
      this Primitive's Model. This constructor assumes that the given indices
      are valid (or will be valid by the time this Primitive gets rendered).

      @param indices  array of vertex and color indices to place in this {@code TriangleFan}
   */
   public TriangleFan(int... indices)
   {
      super(indices);
   }


   /**
      Implement the {@link #makeCopy()} method from the {@link Primitive} abstract base class.
      <p>
      Construct a deep copy of this {@code TriangleFan} object.

      @return a {@code TriangleFan} object that is a deep copy of this {@code TriangleFan} object
   */
   @Override
   public TriangleFan makeCopy() // a kind of "copy constructor"
   {
      TriangleFan result = new TriangleFan();
      for (int i = 0; i < this.vIndexList.size(); i++)
      {
         result.addIndices(vIndexList.get(i),
                           cIndexList.get(i),
                           cIndexList2.get(i));
      }
      return result;
   }


   /**
      Give the triangles of this {@code TriangleFan} the back face colors
      indexed by the given color indices.
      <p>
      NOTE: This method does not put a Color object into this TriangleFan's
      Model. This method assumes that the given index is valid (or will be
      valid by the time this TriangleFan gets rendered).

      @param c0  integer index into a {@link Model}'s color list
      @param c1  integer index into a {@link Model}'s color list
      @param c2  integer index into a {@link Model}'s color list
   */
   @Override
   public void setBackFaceColorIndex(int c0, int c1, int c2)
   {
      cIndexList2.set(0, c0);

      int[] c = {c2, c1};
      for (int i = 1; i < cIndexList2.size(); ++i)
      {
         cIndexList2.set(i, c[i % 2]);
      }
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code TriangleFan} object
   */
   @Override
   public String toString()
   {
      String result =  "TriangleFan: ([";
      for (int i = 0; i < vIndexList.size() - 1; ++i)
      {
         result += vIndexList.get(i) + ", ";
      }
      result += vIndexList.get(vIndexList.size() - 1) + "], [";
      for (int i = 0; i < cIndexList.size() - 1; ++i)
      {
         result += cIndexList.get(i) + ", ";
      }
      result += cIndexList.get(cIndexList.size() - 1) + "], [";
      for (int i = 0; i < cIndexList2.size() - 1; ++i)
      {
         result += cIndexList2.get(i) + ", ";
      }
      result += cIndexList2.get(cIndexList2.size() - 1) + "])\n";
      return result;
   }
}
