/*
 * Renderer 3. The MIT License.
 * Copyright (c) 2022 rlkraft@pnw.edu
 * See LICENSE for details.
*/

import renderer.scene.*;
import renderer.scene.primitives.*;
import renderer.scene.util.DrawSceneGraph;
import renderer.framebuffer.*;
import renderer.pipeline.*;

import java.awt.Color;

/**
   Compile and run this program. Look at its output, both
   in the console window and in the PPM file that it creates.
<p>
   This program draws four lines, two in each of two viewports.
   Each viewport has a different background color.
<p>
   In each viewport you can turn on or turn off anti-aliasing
   (anti-aliasing is currently turned on in each viewport).
<p>
   In each viewport you can also turn on or turn off clipping.
<p>
   To see the anti-aliased pixels, it helps to use a pixel
   magnifying utility. You can magnify the output image
   using IrfanView, but a pixel magnifying utility will
   work better.
<p>
   Notice that the console logging shows all the details
   of how each pixel is anti-aliased.
*/
public class LinesInTwoViewports_R3
{
   public static void main(String[] args)
   {
      final Scene scene = new Scene("LinesInTwoViewports_R3");
      final Model model = new Model("lines");
      scene.addPosition(new Position(model, "p0"));

      model.addVertex(new Vertex(-0.7, -0.1, -1),
                      new Vertex( 1.7,  0.2, -1),
                      new Vertex( 0.3,  1.9, -1));

      model.addColor(Color.blue,
                     Color.red);

      model.addPrimitive(new LineSegment(0, 1, 0),   // blue, slope = 1/8
                         new LineSegment(0, 2, 1));  // red,  slope = 2

      final int widthFB  = 300;
      final int heightFB = 200;
      final FrameBuffer fb = new FrameBuffer(widthFB, heightFB, Color.gray);

      final int widthVP  = 100;
      final int heightVP = 100;
      FrameBuffer.Viewport vp1 = fb.new Viewport( 50, 50, widthVP, heightVP, Color.white);
      FrameBuffer.Viewport vp2 = fb.new Viewport(150, 50, widthVP, heightVP, Color.black);
      vp1.clearVP();
      vp2.clearVP();

      scene.debug = true;
      Rasterize.debug = true;
      Clip.debug = true;

      Rasterize.doGamma = false; // Try turning gamma correction on.
      Rasterize.doAntiAliasing = true;

      Pipeline.render(scene, vp1);

      Rasterize.doGamma = false; // Try turning gamma correction on.
      Rasterize.doAntiAliasing = true;

      Pipeline.render(scene, vp2);

      fb.dumpFB2File("LinesInTwoViewports_R3.ppm");
      System.out.println("Saved " + "LinesInTwoViewports_R3.ppm");
   }
}
