/*
 * Renderer 3. The MIT License.
 * Copyright (c) 2022 rlkraft@pnw.edu
 * See LICENSE for details.
*/

import renderer.scene.*;
import renderer.scene.primitives.*;
import renderer.framebuffer.FrameBuffer;
import renderer.pipeline.*;

import java.awt.Color;

/**
   Here is a picture of a line segment, from vertex v0
   to vertex v1, that needs to be clipped four times
   before we get the final clipped line segment from
   vertex v4 to vertex v5. Remember that the clipping
   algorithm clips first on the edge {@code x = 1},
   then on {@code x = -1}, then on {@code y = 1}, and
   lastly on {@code y = -1}.
<pre>{@code
                              v0
                              +
                          |  /
                          | /
                          |/
                          + v2
      x = -1             /|
        |               / |
        |              /  |
     ---+-------------+---+----- y = 1
        |            /v4  |
        |           /     |
        |          /      |
        |         /       |
        |        /        |
        |       /         |
        |      /          |
        |     /           |
        |    /            |
     ---+---+-------------+----- y = -1
        |  /v5            |
        | /               |
        |/                |
        + v3            x = 1
       /|
      / |
     +
    v1
}</pre>
<p>
   When the clipping algorithm is finished, it will have
   added four new vertices and four new colors to the vertex
   and color lists of this scene's model. But the single line
   segment in the model will only use two of the six vertices
   and two of the six colors in the model's vertex and color
   lists.
<p>
   The clipping algorithm does not try to remove vertices like
   v0, v1, v2 and v3 which are no longer being used. The
   vertices v0 and v1 cannot be removed because they might
   still be used by other line segments in the model. And it
   is not worth the effort to try and keep track of vertices
   like v2 and v3 which are created by the clipping algorithm
   but end up not being used in the final clipped line segment.
<p>
   When you run this program you can see the debugging output
   from the clipping algorithm and you can see the new vertices
   and colors being added to the model's vertex and color lists.
*/
public class ShowExtremeClippingExample_R3
{
   public static void main(String[] args)
   {
      final Scene scene = new Scene("ShowExtremeClippingExample_R3");
      final Model model = new Model("extreme-clipping-example");
      scene.addPosition(new Position(model,
                                     "p0",
                                     new Vector(0, 0, -1)));

      model.addVertex(new Vertex( 1.25,  1.5, 0),
                      new Vertex(-1.25, -1.5, 0));

      model.addColor(new Color(255, 0, 0),
                     new Color(0, 0, 255));

      model.addPrimitive(new LineSegment(0, 1, 0, 1));

      System.out.println(model);

      final int width  = 300;
      final int height = 300;
      final FrameBuffer fb = new FrameBuffer(width, height, Color.white);

      Rasterize.doAntiAliasing = false;
      Rasterize.doGamma = true;

      scene.debug = true;
      Clip.debug = true;
      Rasterize.debug = false;

      Pipeline.render(scene, fb);
      fb.dumpFB2File("ShowExtremeClippingExample_R3.ppm");
      System.out.println("Saved " + "ShowExtremeClippingExample_R3.ppm");
   }
}

/*
Exercise: Suppose that the order of the edges used by the
clipping algorithm is
   x =  1,
   y = -1,
   x = -1,
   y =  1.
How many times would the line in the above example be clipped?

Using this second order for the edges in the clipping algorithm,
find an example of a line that will be clipped four times.
*/
