/*
 * Renderer Models. The MIT License.
 * Copyright (c) 2022 rlkraft@pnw.edu
 * See LICENSE for details.
*/

package renderer.models_L;

import renderer.scene.*;
import renderer.scene.primitives.*;
import renderer.scene.util.MeshMaker;

/**
   Create a wireframe model of a sphere centered at the origin.
<p>
   See <a href="https://en.wikipedia.org/wiki/Sphere" target="_top">
                https://en.wikipedia.org/wiki/Sphere</a>
<p>
   A sphere of radius {@code r} is the surface of revolution generated by
   revolving a half-circle in the xy-plane with radius {@code r} and center
   {@code (0,0,0)} around the y-axis.
<p>
   Here are parametric equations for the right half-circle in the xy-plane with
   radius {@code r} and center {@code (0,0,0)}, parameterized from the top down.
   <pre>{@code
      x(phi) = r * sin(phi)  \
      y(phi) = r * cos(phi)   |-  0 <= phi <= PI
      z(phi) = 0             /
   }</pre>
   Here is the 3D rotation matrix that rotates around the y-axis
   by {@code theta} radians, {@code 0 <= theta <= 2*PI}
   <pre>{@code
      [ cos(theta)   0   sin(theta)]
      [     0        1       0     ]
      [-sin(theta)   0   cos(theta)]
   }</pre>
   If we multiply the rotation matrix with the half-circle
   parameterization, we get a parameterization of the sphere.
   <pre>{@code
      [ cos(theta)   0   sin(theta)]   [r * sin(phi)]
      [     0        1       0     ] * [r * cos(phi)]
      [-sin(theta)   0   cos(theta)]   [     0      ]

      = ( r * sin(phi) * cos(theta).    \
          r * cos(phi),                  |- 0<=theta<=2*PI,  0<=phi<=PI
         -r * sin(phi) * sin(theta) )   /
   }</pre>
<p>
   See:<br>
     <a href="https://en.wikipedia.org/wiki/Sphere#Equations_in_three-dimensional_space" target="_top">
              https://en.wikipedia.org/wiki/Sphere#Equations_in_three-dimensional_space</a>

   @see SphereSector
*/
public class Sphere extends Model implements MeshMaker
{
   public final double r;
   public final int n;
   public final int k;

   /**
      Create a sphere of radius 1 centered at the origin.
   */
   public Sphere( )
   {
      this(1, 15, 16);
   }


   /**
      Create a sphere of radius {@code r} centered at the origin.

      @param r  radius of the sphere
   */
   public Sphere(final double r)
   {
      this(r, 15, 16);
   }


   /**
      Create a sphere of radius {@code r} centered at the origin.
   <p>
      The last two parameters determine the number of half circles
      of longitude and the number of circles of latitude in the model.
   <p>
      If there are {@code k} half circles of longitude, then each circle
      of latitude will have {@code k} line segments.
      If there are {@code n} circles of latitude, then each half circle
      of longitude will have {@code n+1} line segments.
   <p>
      There must be at least three half circles of longitude and
      at least one circle of latitude.

      @param r  radius of the sphere
      @param n  number of circles of latitude
      @param k  number of half circles of longitude
      @throws IllegalArgumentException if {@code n} is less than 1
      @throws IllegalArgumentException if {@code k} is less than 3
   */
   public Sphere(final double r, final int n, final int k)
   {
      super(String.format("Sphere(%.2f,%d,%d)", r, n, k));

      if (n < 1)
         throw new IllegalArgumentException("n must be greater than 0");
      if (k < 3)
         throw new IllegalArgumentException("k must be greater than 2");

      this.r = r;
      this.n = n;
      this.k = k;

      // Create the sphere's geometry.

      final double deltaPhi = Math.PI / (n + 1),
                   deltaTheta = (2 * Math.PI) / k;

      // An array of vertices to be used to create line segments.
      final Vertex[][] v = new Vertex[n][k];

      // Create all the vertices.
      for (int j = 0; j < k; ++j) // choose an angle of longitude
      {
         final double c1 = Math.cos(j * deltaTheta),
                      s1 = Math.sin(j * deltaTheta);
         for (int i = 0; i < n; ++i) // choose an angle of latitude
         {
            final double c2 = Math.cos(deltaPhi + i * deltaPhi),
                         s2 = Math.sin(deltaPhi + i * deltaPhi);
            v[i][j] = new Vertex( r * s2 * c1,
                                  r * c2,
                                 -r * s2 * s1 );
         }
      }
      final Vertex northPole = new Vertex(0,  r, 0),
                   southPole = new Vertex(0, -r, 0);

      // Add all of the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < k; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex(northPole,
                southPole);
      final int northPoleIndex = n * k,
                southPoleIndex = northPoleIndex + 1;

      // Create the horizontal circles of latitude around the sphere.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < k - 1; ++j)
         {  //                                v[i][j]       v[i][j+1]
            addPrimitive(new LineSegment( (i * k) + j,  (i * k) + (j+1) ));
         }
         // close the circle
         addPrimitive(new LineSegment( (i * k) + (k-1), (i * k) + 0 ));
      }  //                                v[i][k-1]        v[i][0]

      // Create the vertical half-circles of longitude from north to south pole.
      for (int j = 0; j < k; ++j)
      {  //                                                v[0][j]
         addPrimitive(new LineSegment( northPoleIndex, (0 * k) + j ));

         for (int i = 0; i < n - 1; ++i)
         {  //                                v[i][j]      v[i+1][j]
            addPrimitive(new LineSegment( (i * k) + j, ((i+1) * k) + j ));
         }
         //                                 v[n-1][j]
         addPrimitive(new LineSegment( ((n-1) * k) + j, southPoleIndex ));
      }
   }



   // Implement the MeshMaker interface (three methods).
   @Override public int getHorzCount() {return n;}

   @Override public int getVertCount() {return k;}

   @Override
   public Sphere remake(final int n, final int k)
   {
      return new Sphere(this.r, n, k);
   }
}//Sphere
