/*
 * Renderer 3. The MIT License.
 * Copyright (c) 2022 rlkraft@pnw.edu
 * See LICENSE for details.
*/

package renderer.pipeline;

import renderer.scene.*;
import renderer.scene.primitives.*;
import static renderer.pipeline.PipelineLogger.*;

import java.util.Optional;

/**
   Clip a (projected) {@link Point} that sticks out
   of the camera's view rectangle in the image plane.
*/
public class Clip_Point
{
   /**
      If the {@link Vertex} used by the {@link Point} sticks out
      of the camera's view rectangle, then have the {@link Point}
      deleted from the model's primitive list.

      @param model  {@link Model} that the {@link Point} {@code pt} comes from
      @param pt     {@link Point} to be clipped
      @return the clipped version of {@code pt} wrapped in an {@link Optional} object
   */
   public static Optional<Primitive> clip(final Model model, final Point pt)
   {
      final Optional<Primitive> result;  // blank final

      // Make local copies of several values.
      final int vIndex = pt.vIndexList.get(0);
      final Vertex v = model.vertexList.get(vIndex);

      final double x = v.x,
                   y = v.y;

      // 1. Check for trivial accept.
      if ( Math.abs( x ) <= 1
        && Math.abs( y ) <= 1 )
      {
         result = Optional.of(pt);  // better than "return pt"
      }
      // 2. Trivial delete.
      else
      {
         result = Optional.empty();  // better than "return null"
      }

      return result;
   }



   // Private default constructor to enforce noninstantiable class.
   // See Item 4 in "Effective Java", 3rd Ed, Joshua Bloch.
   private Clip_Point() {
      throw new AssertionError();
   }
}
