/*
 * Renderer 5. The MIT License.
 * Copyright (c) 2022 rlkraft@pnw.edu
 * See LICENSE for details.
*/

import renderer.scene.*;
import renderer.models_L.Disk;
import renderer.pipeline.*;
import renderer.framebuffer.*;

import java.awt.Color;

/**
   This example demonstrates a form of "letter boxing".
<p>
   This program uses Java's resize window event handler and
   the FrameBuffer's viewport to solve the problem of changing
   aspect ratios demonstrated by the program Circle_v1_Distort.java
<p>
   This program always uses a square viewport that is as large
   as possible within the framebuffer. A square viewport matches
   the shape of the camera's view rectangle, so the image in the
   viewport will not be a distortion of the image in the camera's
   view rectangle.
<p>
   This program has nine modes for how it places the square
   viewport within the framebuffer.
<p>
   Mode 1 places the viewport in the upper left-hand corner
   of the framebuffer.
<p>
   Mode 2 places the viewport at the center of the top edge
   of the framebuffer.
<p>
   Mode 3 places the viewport in the upper right-hand corner
   of the framebuffer.
<p>
   Mode 4 places the viewport at the center of the right edge
   of the framebuffer.
<p>
   Mode 5 places the viewport in the lower right-hand corner
   of the framebuffer.
<p>
   Mode 6 places the viewport at the center of the bottom edge
   of the framebuffer.
<p>
   Mode 7 places the viewport in the lower left-hand corner
   of the framebuffer.
<p>
   Mode 8 places the viewport at the center of the left edge
   of the framebuffer.
<p>
   Mode 9 places the viewport at the center of the framebuffer.
*/
@SuppressWarnings("serial")
public class Circle_v2_Letterbox extends Circle_v0_Abstract
{
   /**
      This constructor instantiates the Scene object
      and initializes it with appropriate geometry.
      Then this constructor instantiates the GUI.
   */
   public Circle_v2_Letterbox()
   {
      super("Renderer 5 - Letterbox");
      print_help_message();
   }


   /**
      Get in one place the code to set up
      the view volume and the viewport.
   */
   @Override protected void setupViewing()
   {
      // Get the size of the FrameBuffer.
      final FrameBuffer fb = fbp.getFrameBuffer();
      final int wFB = fb.getWidthFB();
      final int hFB = fb.getHeightFB();

      // Compute the largest possible dimension for a square viewport.
      final int dVP = Math.min(wFB, hFB);

      // Compute a displacement for the viewport within the framebuffer.
      final int hOffset = (hFB < wFB) ? (wFB - hFB) / 2 : 0;  // landscape : portrait
      final int vOffset = (wFB < hFB) ? (hFB - wFB) / 2 : 0;  // portrait  : lanscape
/*
      final int hOffset = (hFB < wFB) ? (wFB - hFB) / 2 :       0        ;
      final int vOffset = (hFB < wFB) ?      0          : (hFB - wFB) / 2;
      //                                 landscape           portrait
*/
      // Fit the largest possible viewport, with aspect ratio 1,
      // in the framebuffer, located at the framebuffer's:
      if (mode == 1)
      {
         fb.setViewport(0,         0,         dVP, dVP); // 1. upper left-hand corner
      }
      else if (mode == 2)
      {
         fb.setViewport(hOffset,   0,         dVP, dVP); // 2. center of the top edge
      }
      else if (mode == 3)
      {
         fb.setViewport(wFB - dVP, 0,         dVP, dVP); // 3. upper right-hand corner
      }
      else if (mode == 4)
      {
         fb.setViewport(wFB - dVP, vOffset,   dVP, dVP); // 4. center of the right edge
      }
      else if (mode == 5)
      {
         fb.setViewport(wFB - dVP, hFB - dVP, dVP, dVP); // 5, lower right-hand corner
      }
      else if (mode == 6)
      {
         fb.setViewport(hOffset,   hFB - dVP, dVP, dVP); // 6. center of the bottom edge
      }
      else if (mode == 7)
      {
         fb.setViewport(0,         hFB - dVP, dVP, dVP); // 7. lower left-hand corner
      }
      else if (mode == 8)
      {
         fb.setViewport(0,         vOffset,   dVP, dVP); // 8. center of the left edge
      }
      else if (mode == 9)
      {
         fb.setViewport(hOffset,   vOffset,   dVP, dVP); // 9. center of the framebuffer
      }

      super.setupViewing();

    //fb.setViewport(0,          0,          dVP,  dVP); // 1. upper left-hand corner
    //fb.setViewport(hOffset,    0,          dVP,  dVP); // 2. top, center
    //fb.setViewport(wFB - dVP,  0,          dVP,  dVP); // 3. upper right-hand corner
    //fb.setViewport(wFB - dVP,  vOffset,    dVP,  dVP); // 4. right, center
    //fb.setViewport(wFB - dVP,  hFB - dVP,  dVP,  dVP); // 5, lower right-hand corner
    //fb.setViewport(hOffset,    hFB - dVP,  dVP,  dVP); // 6. bottom, center
    //fb.setViewport(0,          hFB - dVP,  dVP,  dVP); // 7. lower left-hand corner
    //fb.setViewport(0,          vOffset,    dVP,  dVP); // 8. left, center
    //fb.setViewport(hOffset,    vOffset,    dVP,  dVP); // 9. centered in the framebuffer


    //fb.setViewport(0,          0,          dVP,  dVP); // upper left-hand corner
    //fb.setViewport(hOffset,    0,          dVP,  dVP); // top, center
    //fb.setViewport(wFB - dVP,  0,          dVP,  dVP); // upper right-hand corner
    //fb.setViewport(0,          vOffset,    dVP,  dVP); // left, center
    //fb.setViewport(hOffset,    vOffset,    dVP,  dVP); // centered in the framebuffer
    //fb.setViewport(wFB - dVP,  vOffset,    dVP,  dVP); // right, center
    //fb.setViewport(0,          hFB - dVP,  dVP,  dVP); // lower left-hand corner
    //fb.setViewport(hOffset,    hFB - dVP,  dVP,  dVP); // bottom, center
    //fb.setViewport(wFB - dVP,  hFB - dVP,  dVP,  dVP); // lower right-hand corner


    //fb.setViewport(0,          0,          dVP,  dVP); // upper left-hand corner
    //fb.setViewport(0,          vOffset,    dVP,  dVP); // left, center
    //fb.setViewport(0,          hFB - dVP,  dVP,  dVP); // lower left-hand corner
    //fb.setViewport(hOffset,    0,          dVP,  dVP); // top, center
    //fb.setViewport(hOffset,    vOffset,    dVP,  dVP); // centered in the framebuffer
    //fb.setViewport(hOffset,    hFB - dVP,  dVP,  dVP); // bottom, center
    //fb.setViewport(wFB - dVP,  0,          dVP,  dVP); // upper right-hand corner
    //fb.setViewport(wFB - dVP,  vOffset,    dVP,  dVP); // right, center
    //fb.setViewport(wFB - dVP,  hFB - dVP,  dVP,  dVP); // lower right-hand corner


    //fb.setViewport(0,          0,          dVP,  dVP); // upper left-hand corner
    //fb.setViewport(wFB - dVP,  0,          dVP,  dVP); // upper right-hand corner
    //fb.setViewport(wFB - dVP,  hFB - dVP,  dVP,  dVP); // lower right-hand corner
    //fb.setViewport(0,          hFB - dVP,  dVP,  dVP); // lower left-hand corner
    //fb.setViewport(hOffset,    0,          dVP,  dVP); // top, center
    //fb.setViewport(wFB - dVP,  vOffset,    dVP,  dVP); // right, center
    //fb.setViewport(hOffset,    hFB - dVP,  dVP,  dVP); // bottom, center
    //fb.setViewport(0,          vOffset,    dVP,  dVP); // left, center
    //fb.setViewport(hOffset,    vOffset,    dVP,  dVP); // centered in the framebuffer
   }


   /**
      Create an instance of this class which has
      the affect of creating the GUI application.
   */
   public static void main(String[] args)
   {
      // We need to call the program's constructor in the
      // Java GUI Event Dispatch Thread, otherwise we get a
      // race condition between the constructor (running in
      // the main() thread) and the very first ComponentEvent
      // (running in the EDT).
      javax.swing.SwingUtilities.invokeLater(
         () -> new Circle_v2_Letterbox()
      );
   }
}
