/*
   This ast2infix() method takes an Abstract Syntax Tree
   for Language-14 and converts the tree into a string that
   contains a "C like" syntax, including infix notation for
   the arithmetic and boolean expressions.

   ast2infix() uses knowledge about operator precedence (in infix
   notation) to use the minimun number of parentheses.
*/

public class AST2infix
{
   public static String ast2infix(Tree tree)
   {
      return ast2infix( tree, 0, true );
   }


   private static String ast2infix(Tree tree, int indentation, boolean inline)
   {
      String indent = "";
      for (int i = 0; i < indentation; i++)
         indent += " ";

      String node = tree.getElement();
      String result = "";


      if ( node.equals("prog") )
      {
         result += "Program\n{\n";
         indentation += 3;

         // process each definition
         for (int i = 0; i < tree.degree(); i++)
         {
            result += ast2infix( tree.getSubTree(i), indentation, false );
            result += "\n";
         }

         result += indent + "}";
      }
      else if ( node.equals("var") )
      {
         String name = tree.getSubTree(0).getElement();

         String rhs = tree.getSubTree(1).getElement();

         if ( rhs.equals("lambda") )  // function definition
         {
            result += name + "(";
            // get the formal parameter list
            if ( tree.getSubTree(1).degree() > 1 )
            {
               result += tree.getSubTree(1).getSubTree(0).getElement();
               for (int i = 1; i < tree.getSubTree(1).degree()-1; i++)
               {
                  result += ", " + tree.getSubTree(1).getSubTree(i).getElement();
               }
            }
            result += ")\n";

            // get the body of the function
            Tree expr = tree.getSubTree(1).getSubTree(tree.getSubTree(1).degree()-1);
            if ( ! expr.getElement().equals("begin") )
            {
               result += indent + "{\n";
               indentation += 3;
            }
            result += ast2infix( expr, indentation, false );
            if ( ! expr.getElement().equals("begin") )
            {
               result += ";\n" + indent + "}";
            }
            result += "\n";
         }
         else // variable definition
         {
            indentation += 7 + name.length();
            result += "var " + name + " = "
                             + ast2infix( tree.getSubTree(1), indentation, true );
         }
      }
      else if ( node.equals("lambda") )
      {
         // get the formal parameter list
         if ( tree.degree()==2 ) // exactly one formal parameter
         {
            result += tree.getSubTree(0).getElement();
         }
         else // zero, or more than one formal parameter
         {
            result += "(";
            if ( tree.degree() >= 3 )  // two or more formal parameters
            {
               result += tree.getSubTree(0).getElement();
               for (int i = 1; i < tree.degree()-1; i++)
               {
                  result += ", " + tree.getSubTree(i).getElement();
               }
            }
            result += ")";
         }
         // get the body of the function
         result += "->";
         Tree expr = tree.getSubTree(tree.degree()-1);
         result += ast2infix( expr, indentation, true );
      }
      else if ( node.equals("apply") )
      {
         if ( tree.getSubTree(0).getElement().equals("lambda") )
         { // the anonymous lambda expression case
            result += "("+ast2infix( tree.getSubTree(0), indentation, true )+")";
         }
         else // the named function case
         {
            String name = tree.getSubTree(0).getElement();
            result += name;
         }

         indentation += 3;
         // get the actual parameter list
         result += "(";
         if ( tree.degree() > 1 )
         {
            result += ast2infix( tree.getSubTree(1), indentation, true );
            for (int i = 2; i < tree.degree(); i++)
            {
               result += ", " + ast2infix( tree.getSubTree(i), indentation, true );
            }
         }
         result += ")";
      }
      else if ( node.equals("print") )
      {
         indentation += 8;
         result += "print(" + ast2infix( tree.getSubTree(0), indentation, true ) + ")";
      }
      else if ( node.equals("if") )
      {
         indentation += 7;
         result += "if (" + ast2infix( tree.getSubTree(0), indentation, true ) + ")\n";
         indentation -= 4;
         result += ast2infix( tree.getSubTree(1), indentation, false ) + "\n";
         result += indent + "else\n";
         result += ast2infix( tree.getSubTree(2), indentation, false );
      }
      else if ( node.equals("while") )
      {
         indentation += 7;
         result += "while (" + ast2infix( tree.getSubTree(0), indentation, true ) + ")\n";

         if ( tree.getSubTree(1).getElement().equals("begin") )
         {
            indentation -= 7;
            result += ast2infix( tree.getSubTree(1), indentation, false );
         }
         else
         {
            indentation -= 4;
            result += ast2infix( tree.getSubTree(1), indentation, false );
         }
      }
      else if ( node.equals("begin") )
      {
         result += "{\n";
         indentation += 3;

         // process each expression
         for(int i = 0; i < tree.degree(); i++)
         {
            result += ast2infix( tree.getSubTree(i), indentation, false );
            if ( tree.getSubTree(i).getElement().equals("begin") )
               result += "\n";
            else
               result += ";\n";
         }

         result += indent + "}";
      }
      else if ( node.equals("set") )
      {
         indentation += 3 + tree.getSubTree(0).getElement().length();
         result += tree.getSubTree(0).getElement()
                   + " = " + ast2infix( tree.getSubTree(1), indentation, true );
      }
      else if ( node.equals("list") )
      {
         indentation += 2;

         result += "[";
         for (int i = 0; i < tree.degree(); i++)
            result += " " + ast2infix( tree.getSubTree(i), indentation, true );
         result += " ]";
      }
      else if ( node.equals("sym") )
      {
         // put a "quote" in front of the symbol, as in Lisp
         result += "'"+tree.getSubTree(0).getElement();
      }
      else if ( tree.degree() == 0 )  // constant term
      {
         result += node;
      }

      if (!inline)
         result = indent + result;

      return result;
   }//ast2infix()


   private static String builtin(Tree tree, int indentation, boolean inline)
   {
      String result = "";

      String op = tree.getSubTree(0).getElement();

      if ( tree.degree() == 2 )  // unary operator
      {
         if ( tree.getSubTree(0).degree() >= 2 )
         {
            result += op + "(" + ast2infix( tree.getSubTree(0) ) + ")";
         }
         else
         {
            result += op + ast2infix( tree.getSubTree(0) );
         }
      }
      else if ( tree.degree() == 3 )  // binary operator
      {
         int middlePrecedence = precedence( op );
         int leftPrecedence = precedence( tree.getSubTree(1).getElement() );

         if ( leftPrecedence > middlePrecedence )
         {
            result += "(" + ast2infix( tree.getSubTree(1) ) + ")";
         }
         else
         {
            result += ast2infix( tree.getSubTree(1) );
         }

         result += " " + op + " ";

         int rightPrecedence = precedence( tree.getSubTree(2).getElement() );

         if ( rightPrecedence >= middlePrecedence )
         {
            result += "(" + ast2infix( tree.getSubTree(2) ) + ")";
         }
         else
         {
            result += ast2infix( tree.getSubTree(2) );
         }
      }
      else  // operators with 3 or more operands
      {
         int operatorPrecedence = precedence( op );
         int leftPrecedence = precedence( tree.getSubTree(1).getElement() );

         if ( leftPrecedence > operatorPrecedence )
         {
            result += "(" + ast2infix( tree.getSubTree(1) ) + ")";
         }
         else
         {
            result += ast2infix( tree.getSubTree(1) );
         }

         for (int i = 2; i < tree.degree(); i++)
         {
            result += " " + op + " ";

            int rightPrecedence = precedence( tree.getSubTree(i).getElement() );

            if ( rightPrecedence >= operatorPrecedence )
            {
               result += "(" + ast2infix( tree.getSubTree(i) ) + ")";
            }
            else
            {
               result += ast2infix( tree.getSubTree(i) );
            }
         }
      }

      return result;
   }//builtin()


   // http://www.java-tips.org/java-se-tips/java.lang/what-is-java-operator-precedence.html
   // http://www.cs.princeton.edu/introcs/11precedence/
   private static int precedence(String op)
   {
      int result = 0;  // "highest" precedence (i.e., constants)

      if ( op.equals("!") )
      {
         result = 2;
      }
      else if ( op.equals("^") )
      {
         result = 3;
      }
      else if ( op.equals("*")
             || op.equals("/")
             || op.equals("%") )
      {
         result = 4;
      }
      else if ( op.equals("+")
             || op.equals("-") )
      {
         result = 5;
      }
      else if ( op.equals("<")
             || op.equals(">")
             || op.equals("<=")
             || op.equals(">=") )
      {
         result = 7;
      }
      else if ( op.equals("==")
             || op.equals("!=") )
      {
         result = 8;
      }
      else if ( op.equals("&&") )
      {
         result = 11;
      }
      else if ( op.equals("||") )
      {
         result = 12;
      }
      else if ( op.equals("var") || op.equals("set") )
      {
         result = 14;
      }

      return result;
   }

}//AST2infix