/**
   A Environment object holds <variable,value> pairs and
   has a link to its nesting Environment object (its outer scope).

   For this language, in order to implement function parameters,
   we must have a single namespace for all kinds of variables,
   those that name integers, those that name booleans, and those
   that name functions. Therefore, for this language, the "value"
   that can be stored in a <variable,value> pair can have the
   types "int", "bool", "closure", "builtin", "list", or "sym".
*/
import java.util.ArrayList;

public class Environment
{
   private ArrayList<String> variables;
   private ArrayList<Value>  values;
   private Environment nestingLink;


   // Constructors
   public Environment()
   {
      variables = new ArrayList<String>();
      values    = new ArrayList<Value>();
      nestingLink = null;
   }

   public Environment(Environment env)
   {
      variables = new ArrayList<String>();
      values    = new ArrayList<Value>();
      nestingLink = env;
   }

   /**
      Add a <variable, value> pair to this environment object.
   */
   public void add(String variable, Value value)
   {
      variables.add(variable);
         values.add(value);
   }

   /**
      Look up variable in the environment chain and
      return its associated value.

      Throws an Error if variable is not found.
   */
   public Value lookUp(String variable)
   {
      int i;
      for (i=0; i < variables.size(); i++)
         if (variable.equals(variables.get(i)))
            break;

      if ( i < variables.size() )
      {
         return values.get(i);
      }
      else
      {
         if (nestingLink == null)
            throw new Error("undefined variable: " + variable);
         else
            return nestingLink.lookUp(variable);
      }
   }

   /**
      Look up variable in the environment chain.
      Return true if the variable is in the chain,
      otherwise return false.
   */
   public boolean defined(String variable)
   {
      int i;
      for (i=0; i < variables.size(); i++)
         if (variable.equals(variables.get(i)))
            break;

      if ( i < variables.size() )
      {
         return true;
      }
      else
      {
         if (nestingLink == null)
            return false;
         else
            return nestingLink.defined(variable);
      }
   }


   /**
      Look up variable in this environment object.
      Return true if the variable is in this object,
      otherwise return false.
   */
   public boolean definedLocal(String variable)
   {
      int i;
      for (i=0; i < variables.size(); i++)
         if (variable.equals(variables.get(i)))
            break;

      if ( i < variables.size() )
      {
         return true;
      }
      else
      {
         return false;
      }
   }


   /**
      Update the value associated with variable in the environment chain.
      Return true if the update is succesfull.
      Return false if variable is not found in the chain.
   */
   public boolean update(String variable, Value value)
   {
      int i;
      for (i=0; i < variables.size(); i++)
         if (variable.equals(variables.get(i)))
            break;

      if ( i < variables.size() )
      {
         values.set(i, value);
         return true;
      }
      else
      {
         if (nestingLink == null)
         {
            return false;
         }
         else
         {
            return nestingLink.update(variable, value);
         }
      }
   }


   /**
      Convert the contents of the environment chain into a string.
      This is mainly for debugging purposes.
   */
   public String toString()
   {
      String result = "";

      if (nestingLink == null)
      {
         result += "[Global Environment";
      }
      else
      {
         result += "[Nested Environment";
      }

      for (int i = 0; i < variables.size(); i++)
      {
         result += "\n " + variables.get(i) + " = " + values.get(i);
      }
      result += "]";

      if (nestingLink == null)
      {
         return result;
      }
      else
      {
         return result + "\n||\n\\/\n" + nestingLink.toString();
      }
   }

}//Environment