/*
   This program evaluates (interprets) an Abstract Syntax Tree (AST)
   that is defined by the following grammar.

       Prog ::= '(' 'prog' Body ')'

       Body ::= Exp+

        Exp ::= Begin | Var | NExp

      Begin ::= '(' 'begin' Body ')'

        Var ::= '(' 'var' VARIABLE NExp ')'

       NExp ::= Set | If | While | And | Or
              | Apply | VARIABLE
              | Lambda | INTEGER | BOOLEAN | List | Symbol

        Set ::= '(' 'set' VARIABLE NExp ')'

         If ::= '(' 'if' NExp Exp Exp ')'

      While ::= '(' 'while' NExp Exp ')'

        And ::= '(' '&&' NExp NExp NExp* ')'

         Or ::= '(' '||' NExp NExp NExp* ')'

      Apply ::= '(' 'apply' NExp NExp* ')'  // function "name" followed by actual parameters

     Lambda ::= '(' 'lambda' VARIABLE* Exp ')'  // formal parameters followed by the body

       List ::= '(' 'list' NExp* ')'   // List doesn't evaluate "symbols"

     Symbol ::= '(' 'sym' SYMBOL ')'

     SYMBOL ::= [a-zA-Z][a-zA-Z0-9]*
   VARIABLE ::= [a-zA-Z][a-zA-Z0-9]*
    INTEGER ::= [0-9]+
    BOOLEAN ::= 'true' | 'false'

This language should also include the following built-in functions.

   'car' Returns the first item from its list parameter
   'cdr' Returns the list formed by removing the first item from its parameter
  'cons' "Constructs" a list by prepending its 1st parameter to its 2nd parameter
'empty?' Returns true if its parameter is an empty list
     '+' Addition of one, or more, integer operands
     '-' Unary negation of an integer operand
     '-' Subtraction of two integer operands
     '*' Multiplication of two, or more, integer operands
     '/' Division of two integer operands
     '%' Remainder of two integer operands
     '^' Exponentiation of two integer operands
     '!' Unary negation of a boolean operand
    '==' Equality of two operands
    '!=' Inequality of two operands
     '<' Less than
     '>' Greater than
    '<=' Less than or equal
    '>=' Greater than or equal
 'print' Print one operand


NOTE: The above language is the preprocessor output for Language-15.
      So it includes the keyword  'apply' in each Apply term.
*/

public class Evaluate
{
   // Global environment for function and variable definitions.
   private static Environment globalEnv;

   public static Value evaluate(Tree tree) throws EvalException
   {
      Value result;

      // Evaluate each Prog term in the File term.
      result = evaluateProg( tree );

      return  result;
   }//evaluate()


   private static Value evaluateProg(Tree tree) throws EvalException
   {
      // Create a new global environment object.
      globalEnv = new Environment();

      // Populate the global environment with the built-in functions.
      globalEnv.add("car",  new Value("builtin", "car") );
      globalEnv.add("cdr",  new Value("builtin", "cdr") );
      globalEnv.add("cons", new Value("builtin", "cons") );
      globalEnv.add("empty?", new Value("builtin", "empty?") );
      globalEnv.add("+", new Value("builtin", "+") );
      globalEnv.add("-", new Value("builtin", "-") );
      globalEnv.add("*", new Value("builtin", "*") );
      globalEnv.add("/", new Value("builtin", "/") );
      globalEnv.add("%", new Value("builtin", "%") );
      globalEnv.add("^", new Value("builtin", "^") );
      globalEnv.add("!", new Value("builtin", "!") );
      globalEnv.add("==", new Value("builtin", "==") );
      globalEnv.add("!=", new Value("builtin", "!=") );
      globalEnv.add("<", new Value("builtin", "<") );
      globalEnv.add(">", new Value("builtin", ">") );
      globalEnv.add("<=", new Value("builtin", "<=") );
      globalEnv.add(">=", new Value("builtin", ">=") );
      globalEnv.add("print", new Value("builtin", "print") );

      Value result = evaluateBody(tree, globalEnv);

      return result;
   }//evaluateProg()


   private static Value evaluateBody(Tree tree, Environment env) throws EvalException
   {
      Value result = null;

      // Evaluate each expression in the Body term.
      // The last expression value is the value of the Body.
      for (int i = 0; i < tree.degree(); i++)
      {
         result = evaluateE( tree.getSubTree(i), env );
      }

      return result;
   }//evaluateBody()


   /**
      Notice that we evaluate an expression using the included environment which
      is a linked list of environment objects, terminating with the global environment.
   */
   private static Value evaluateE(Tree tree, Environment env) throws EvalException
   {
      Value result = null;

      String node = tree.getElement();

      if ( node.equals("begin") )
      {
         result = evaluateBegin(tree, env);
      }
      else if ( node.equals("var") )
      {
         result = evaluateVar(tree, env);
      }
      else
      {
         result = evaluateN(tree, env);
      }

      return result;
   }//evaluateE()


   private static Value evaluateBegin(Tree tree, Environment env) throws EvalException
   {
      // create a new environment "nested" in the previous environment
      Environment localEnv = new Environment(env);

      Value result;

      // evaluate each expression in the body of the Begin (using the nested environment)
      result = evaluateBody(tree, localEnv);

      // Notice that when this evaluateBegin() method returns, the
      // localEnv object becomes a garbage object, so it is implicitly
      // removed from the list of Environment objects.

      return result;
   }//evaluateBegin()


   private static Value evaluateVar(Tree tree, Environment env) throws EvalException
   {
      Value result = new Value("bool", false);  // default return value

      // get the variable
      String variable = tree.getSubTree(0).getElement();

      // get the expression
      Tree expr = tree.getSubTree(1);

      result = evaluateN(expr, env);

      // check if this variable has already been defined
      if ( !env.definedLocal(variable) )
      {
         env.add(variable, result);
      }
      else // this variable is already in the local Environment object
      {
         throw new EvalException("local variable already exisits: " + variable);
      }
//      System.out.println( env + "\n" ); // for debugging purposes

      return result;
   }//evaluateVar()


   private static Value evaluateN(Tree tree, Environment env) throws EvalException
   {
      Value result = null;

      String node = tree.getElement();

      if ( tree.degree() == 0 && !node.equals("list") )
      {
         if ( node.equals("true") || node.equals("false") )
         {
            result = new Value( "bool", node.equals("true") );
         }
         else if ( node.matches("^[0-9][0-9]*") )
         {
            result = new Value( "int", Integer.parseInt( node ) );
         }
         else if ( env.defined(node) )
         {
            result = env.lookUp( node );
         }
         else
         {
            throw new EvalException("undefined variable: " + node);
         }
      }
      else if ( node.equals("set") )
      {
         result = evaluateSet(tree, env);
      }
      else if ( node.equals("if") )
      {
         result = evaluateIf(tree, env);
      }
      else if ( node.equals("while") )
      {
         result = evaluateWhile(tree, env);
      }
      else if ( node.equals("&&") )
      {
         result = evaluateAnd(tree, env);
      }
      else if ( node.equals("||") )
      {
         result = evaluateOr(tree, env);
      }
      else if ( node.equals("apply") )
      {
         result = evaluateApply(tree, env);
      }
      else if ( node.equals("lambda") )
      {
         result = evaluateLambda(tree, env);
      }
      else if ( node.equals("list") )
      {
         result = evaluateList( tree, env );
      }
      else if ( node.equals("sym") )
      {
         result = evaluateSymbol( tree, env );
      }
      else  // shouldn't get here
      {

      }

      return result;
   }//evaluateN()


   private static Value evaluateSet(Tree tree, Environment env) throws EvalException
   {
      // get the variable
      String variable = tree.getSubTree(0).getElement();

      // get, and then evaluate, the expression
      Tree expr = tree.getSubTree(1);
      Value value = evaluateE(expr, env);

      // check if this variable has already been defined
      if ( ! env.defined(variable) )
      {
         throw new EvalException("undefined variable: " + variable);
      }
      else // this variable is already in the environment
      {
         env.update(variable, value);
      }

//      System.out.println( env + "\n" ); // for debugging purposes

      return value;
   }//evaluateSet()


   private static Value evaluateIf(Tree tree, Environment env) throws EvalException
   {
      Value result = null;

      Value conditionalExp = evaluateN( tree.getSubTree(0), env );

      if ( ! conditionalExp.tag.equals("bool") )
      {
         throw new EvalException("illegal boolean expression: "
                                    + AST2infix.ast2infix(tree));
      }

      if ( conditionalExp.valueB )
      {
         result = evaluateE( tree.getSubTree(1), env );
      }
      else
      {
         result = evaluateE( tree.getSubTree(2), env );
      }

      return result;
   }//evaluateIf()


   private static Value evaluateWhile(Tree tree, Environment env) throws EvalException
   {
      Value result;

      Value conditionalExp = evaluateN( tree.getSubTree(0), env );

      if ( ! conditionalExp.tag.equals("bool") )
      {
         throw new EvalException("illegal boolean expression: "
                                    + AST2infix.ast2infix(tree));
      }

      while ( conditionalExp.valueB )
      {
         // evaluate the body of the loop
         result = evaluateE( tree.getSubTree(1), env );

         conditionalExp = evaluateE( tree.getSubTree(0), env );

         if ( ! conditionalExp.tag.equals("bool") )
         {
            throw new EvalException("illegal boolean expression: "
                                       + AST2infix.ast2infix(tree));
         }
      }

      // always return false for a while loop
      result =  new Value( "bool", false );

      return result;
   }//evaluateWhile()


   // Evaluate the boolean "special form" &&
   private static Value evaluateAnd(Tree tree, Environment env) throws EvalException
   {
      boolean result;

      Value value = evaluateN( tree.getSubTree(1), env );
      if ( ! value.tag.equals("bool") )
      {
         throw new EvalException("illegal boolean value: " + value );
      }
      result = value.valueB;

      for (int i = 2; i < tree.degree(); i++)
      {
         if (result)
         {  // get another operand
            value = evaluateN( tree.getSubTree(i), env );
            if (value.tag.equals("bool") )
            {
               result = value.valueB;
            }
            else
            {
               throw new EvalException("illegal value: " + value);
            }
         }
         else
         {
            break;
         }
      }

      return new Value( "bool", result );
   }//evaluateAnd()


   // Evaluate the boolean "special form" ||
   private static Value evaluateOr(Tree tree, Environment env) throws EvalException
   {
      boolean result;

      Value value = evaluateN( tree.getSubTree(1), env );
      if ( ! value.tag.equals("bool") )
      {
         throw new EvalException("illegal boolean value: " + value );
      }
      result = value.valueB;

      for (int i = 2; i < tree.degree(); i++)
      {
         if (! result)
         {  // get another operand
            value = evaluateN( tree.getSubTree(i), env );
            if (value.tag.equals("bool") )
            {
               result = value.valueB;
            }
            else
            {
               throw new EvalException("illegal value: " + value);
            }
         }
         else
         {
            break;
         }
      }

      return new Value( "bool", result );
   }//evaluateOr()


   /**
      This method "applies" a function to actual parameters.
      The function can be either a user defined function (i.e., a closure)
      or a built-in function.
      Bind as many actual parameters as there are formal parameters.
      If there are more actual paramters than formal parameters, ignore
      the extra actual parameters. If there are more formal parameters
      than actual parameters, throw an interpreter exception.
   */
   private static Value evaluateApply(Tree tree, Environment env) throws EvalException
   {
      Value result;

      // Evaluate the first expression, which is supposed to evalute to a
      // function value (either a closure or a builtin function value).
      Value func = evaluateN( tree.getSubTree(0), env );

      // Check for the type of function, builtin or user define (closure)
      if ( func.tag.equals("builtin") )    // built-in function
      {
         String op = func.valueBi;  // name of the built-in function
         result = evaluateBuiltin( op, tree, env );
      }
      else if ( func.tag.equals("closure") )  // user defined function
      {
         // Retrive, from within the Value object,
         // a reference to the closure's IPEP pair.
         IPEP ipepPair = func.valueC;

         // Retrive, from within the IPEP pair,
         // a reference to the closure's function definition
         // (i.e., the function's "lambda expression").
         Tree function = ipepPair.exp;

         // Check that there are at least as many actual parameters
         // as the number of formal parameters.
         if ( function.degree() > tree.degree() )
         {
            throw new EvalException("not enough actual parameters: "
                                     + AST2infix.ast2infix(tree) );
         }

         // Create a new environment "nested" in the environment
         // pointed to by the ipep pair.
         Environment localEnv = new Environment(ipepPair.env);  // NOTE: static scope

         // Bind the actual parameter values to the formal parameters.
         // The binding is done in the new, local environment.
         for (int i = 0; i < function.degree()-1; i++)
         {
            // Evaluate an actual parameter using the original environment.
            Value actualParam = evaluateN( tree.getSubTree(i+1), env );
            // Get a formal parameter
            String formalParam = function.getSubTree(i).getElement();
            // Put a <formal param, actual param> pair in the new environment
            localEnv.add(formalParam, actualParam);
         }
//         System.out.println( localEnv + "\n" ); // for debugging purposes

         // Finally, evaluate the body of the function using the
         // new environment (which contains the binding of the actual
         // parameter value to the function's formal parameter).
         result = evaluateE( function.getSubTree(function.degree()-1), localEnv );
      }
      else  // not a proper function
      {
         throw new EvalException("variable not a function: " + tree.getSubTree(0).getElement());
      }

      return result;
   }//evaluateApply()


   /**
      Evaluate a "lambda expression".
      A lambda expression is a function literal.
      The value of a lambda expression is a closure, an ip-ep pair.
      The "instruction pointer", the ip, points to the function definition.
      The "environment pointer", the ep, points to the environment (the scope)
      in which this lambda expression is being evaluated.
      (NOTE: It is very important to distinguish between "evaluating" a lambda
             expression and "applying" a lambda expression. In fact, you don't
             apply a lambda expression, you apply a closure, which is the result
             of evaluating a lambda expression.)
   */
   private static Value evaluateLambda(Tree tree, Environment env) throws EvalException
   {
      Value result;

      // The value of a lambda expression is a closure.
      result = new Value( "closure", new IPEP(tree,env) );

      return result;
   }//evaluateLambda()


   // Evaluate a list literal
   private static Value evaluateList(Tree tree, Environment env) throws EvalException
   {
      Value result = new Value("list");  // create an empty list

      // fill up the list (notice that we work from right to left in the list)
      for (int i = tree.degree()-1; i >= 0; i--)
      {
         String element = tree.getSubTree(i).getElement();

         if ( tree.getSubTree(i).degree() != 0 || element.equals("list") )
         {
            result.valueL = new ConsCell( evaluateN(tree.getSubTree(i), env), result.valueL );
         }
         else
         {
            Value value;

            if ( element.matches("^[0-9][0-9]*") )
            {
               value = new Value( "int", Integer.parseInt(element) );
            }
            else if ( element.equals("true") || element.equals("false") )
            {
               value = new Value( "bool", element.equals("true") );
            }
            else if ( env.defined(element) )
            {
               value = env.lookUp(element);
            }
            else  // unevaluated variable (a "symbol")
            {
               value = new Value( "sym", element );
            }

            result.valueL = new ConsCell(value, result.valueL);
         }
      }
      return result;
   }//evaluateList()


   // Evaluate a symbol literal
   private static Value evaluateSymbol(Tree tree, Environment env)  throws EvalException
   {
      return new Value( "sym", tree.getSubTree(0).getElement() );
   }//evaluateSymbol()


   /**
      Apply the built-in function with name "op"
      to the actual parameters contained in the
      child nodes of the tree.
   */
   private static Value evaluateBuiltin(String op, Tree tree, Environment env) throws EvalException
   {
      Value result = null;

      if ( op.equals("car") )
      {
         Value list = evaluateE( tree.getSubTree(1), env );
         if ( ! list.tag.equals("list") )
         {
            throw new EvalException("car: argument is not a list" + list);
         }
         if ( list.valueL == null )
         {
            throw new EvalException("car: argument is an empty list" + list);
         }
         else
         {
            result = list.valueL.car();
         }
      }
      else if ( op.equals("cdr") )
      {
         Value list = evaluateE( tree.getSubTree(1), env );
         if ( ! list.tag.equals("list") )
         {
            throw new EvalException("cdr: argument is not a list" + list);
         }
         if ( list.valueL == null )
         {
            throw new EvalException("cdr: argument is an empty list" + list);
         }
         else
         {
            result = new Value( "list", list.valueL.cdr() );
         }
      }
      else if ( op.equals("cons") )
      {
         Value list = evaluateE( tree.getSubTree(2), env );
         if ( ! list.tag.equals("list") )
         {
            throw new EvalException("cons: 2nd argument is not a list" + list);
         }
         else
         {
            Value car = evaluateE( tree.getSubTree(1), env );
            result = new Value( "list", new ConsCell(car, list.valueL) );
         }
      }
      else if ( op.equals("empty?") )
      {
         Value list = evaluateE( tree.getSubTree(1), env );
         if ( ! list.tag.equals("list") )
         {
            throw new Error("empty?: argument is not a list" + list);
         }
         else
         {
            result = new Value( "bool", list.valueL == null );
         }
      }
      else if ( op.equals("+")            // arithmetic operators
        || op.equals("-")
        || op.equals("*")
        || op.equals("/")
        || op.equals("%")
        || op.equals("^") )
      {
         result = evaluateA(op, tree, env);
      }
      else if ( op.equals("!") )
      {
         result = evaluateNot(tree, env);
      }
      else if ( op.equals("==") ||    // equality operators
                op.equals("!=") )
      {
         result = evaluateEq(op, tree, env);
      }
      else if ( op.equals("<") ||     // relational operators
                op.equals(">") ||
                op.equals("<=") ||
                op.equals(">=") )
      {
         result = evaluateR(op, tree, env);
      }
      else if ( op.equals("print") )
      {
         result = evaluatePrint(tree, env);
      }
      else  // shouldn't get here
      {
         throw new EvalException("illegal built-in function: " + op);
      }

      return result;
   }//evaluateBuiltin()


   // Evaluate the built-in function "print".
   private static Value evaluatePrint(Tree tree, Environment env) throws EvalException
   {
      Value result = evaluateE( tree.getSubTree(1), env );

      System.out.println( result );

      return result;
   }//evaluatePrint()


   // Evaluate the built-in arithmetic functions
   private static Value evaluateA(String op, Tree tree, Environment env) throws EvalException
   {
      int result = 0;

      Value valueL = null;
      Value valueR = null;
      int resultL = 0;
      int resultR = 0;

      // get first operand
      valueL = evaluateN( tree.getSubTree(1), env );
      if ( valueL.tag.equals("int") )
      {
         resultL = valueL.valueI;
      }
      else
      {
         throw new EvalException("illegal value: " + valueL);
      }
      // get second operand (if there is one)
      if ( tree.degree() >= 3 )
      {
         valueR = evaluateN( tree.getSubTree(2), env );
         if ( valueR.tag.equals("int") )
         {
            resultR = valueR.valueI;
         }
         else
         {
            throw new EvalException("illegal value: " + valueR);
         }
      }

      // evaluate operator
      if ( op.equals("+") )
      {
         if ( tree.degree() == 2 )
         {
            result = resultL;
         }
         else
         {
            result = resultL + resultR;

            for (int i = 3; i < tree.degree(); i++)
            {  // get another operand
               valueR = evaluateN( tree.getSubTree(i), env );
               if ( valueR.tag.equals("int") )
               {  // evalaute operator again
                  result += valueR.valueI;
               }
               else
               {
                  throw new EvalException("illegal value: " + valueR);
               }
            }
         }
      }
      else if ( op.equals("-") )
      {
         if ( tree.degree() == 2 )
         {
            result = -resultL;
         }
         else if ( tree.degree() > 3 )
         {
            throw new EvalException("too many operands for " + op + ": "
                                    + AST2infix.ast2infix(tree));
         }
         else
         {
            result = resultL - resultR;
         }
      }
      else if ( op.equals("*") )
      {
         if ( tree.degree() == 2 )
         {
            throw new EvalException("too few operands for " + op + ": "
                                    + AST2infix.ast2infix(tree));
         }

         result = resultL * resultR;

         for (int i = 3; i < tree.degree(); i++)
         {  // get another operand
            valueR = evaluateN( tree.getSubTree(i), env );
            if ( valueR.tag.equals("int") )
            {  // evaluate operator again
               result *= valueR.valueI;
            }
            else
            {
               throw new EvalException("illegal value: " + valueR);
            }
         }
      }
      else if ( tree.degree() == 2 )
      {
         throw new EvalException("too few operands for " + op + ": "
                                 + AST2infix.ast2infix(tree));
      }
      else if ( tree.degree() > 3 )
      {
         throw new EvalException("too many operands for " + op + ": "
                                 + AST2infix.ast2infix(tree));
      }
      else if ( op.equals("/") )
      {
         result = resultL / resultR;
      }
      else if ( op.equals("%") )
      {
         result = resultL % resultR;
      }
      else if ( op.equals("^") )
      {
         result = (int)Math.pow(resultL, resultR);
      }
      else
      {
         throw new EvalException("invalid arithmetic operator: " + op);
      }

      return new Value( "int", result );
   }//evaluateA()


   // Evaluate the built-in boolean function "!"
   private static Value evaluateNot(Tree tree, Environment env) throws EvalException
   {
      boolean result;

      Value value = evaluateN( tree.getSubTree(1), env );
      if ( ! value.tag.equals("bool") )
      {
         throw new EvalException("illegal boolean value: " + value );
      }
      result =  ! value.valueB;  // negate the boolean result

      return new Value( "bool", result );
   }//evaluateNot()


   // Evaluate the built-in equality functions
   private static Value evaluateEq(String op, Tree tree, Environment env) throws EvalException
   {
      boolean result = false;

      Value valueL = evaluateE( tree.getSubTree(1), env );
      Value valueR = evaluateE( tree.getSubTree(2), env );

      if ( ! valueL.tag.equals(valueR.tag) )
      {
         result = false;
      }
      else if ( valueL.tag.equals("int") )
      {
         result = (valueL.valueI == valueR.valueI);
      }
      else if ( valueL.tag.equals("bool") )
      {
         result = (valueL.valueB == valueR.valueB);
      }
      else
      {
         throw new EvalException("invalid Value tag: " + valueL.tag);
      }

      if ( op.equals("!=") )
      {
         result = ! result;
      }
      else  if ( ! op.equals("==") )
      {
        throw new EvalException("invalid equality operator: " + op);
      }

      return new Value( "bool", result );
   }//evaluateEq()


   // Evaluate the built-in relational functions
   private static Value evaluateR(String op, Tree tree, Environment env) throws EvalException
   {
      boolean result = false;

      Value valueL = evaluateN( tree.getSubTree(1), env );
      Value valueR = evaluateN( tree.getSubTree(2), env );

      if ( ! valueL.tag.equals("int") )
      {
         throw new EvalException("illegal arithmetic value: " + valueL );
      }
      if ( ! valueR.tag.equals("int") )
      {
         throw new EvalException("illegal arithmetic value: " + valueR );
      }

      int resultL = valueL.valueI;
      int resultR = valueR.valueI;

      if ( op.equals("<") )
      {
         result = resultL < resultR;
      }
      else if ( op.equals(">") )
      {
         result = resultL > resultR;
      }
      else if ( op.equals("<=") )
      {
         result = resultL <= resultR;
      }
      else if ( op.equals(">=") )
      {
         result = resultL >= resultR;
      }
      else
      {
         throw new EvalException("invalid relational operator: " + op);
      }

      return new Value( "bool", result );
   }//evaluateR()

}//Evaluate