/*
   This preprocessor consumes strings from the following language:

       Prog ::= '(' 'prog' Body ')'

       Body ::= Exp+

        Exp ::= Begin | Var | NExp

      Begin ::= '(' 'begin' Body ')'

        Var ::= '(' 'var' VARIABLE NExp ')'

       NExp ::= Set | If | While | And | Or
              | Apply | VARIABLE
              | Lambda | INTEGER | BOOLEAN | List | Symbol

        Set ::= '(' 'set' VARIABLE NExp ')'

         If ::= '(' 'if' NExp Exp Exp ')'

      While ::= '(' 'while' NExp Exp ')'

        And ::= '(' '&&' NExp NExp NExp* ')'

         Or ::= '(' '||' NExp NExp NExp* ')'

      Apply ::= '(' NExp NExp* ')'

     Lambda ::= '(' 'lambda' VARIABLE* Exp ')'

       List ::= '(' 'list' NExp* ')'

     Symbol ::= '(' 'sym' SYMBOL ')'

     SYMBOL ::= [a-zA-Z][a-zA-Z0-9]*
   VARIABLE ::= [a-zA-Z][a-zA-Z0-9]*
    INTEGER ::= [0-9]+
    BOOLEAN ::= 'true' | 'false'

   This preprocessor converts those strings into the following language:

       Prog ::= '(' 'prog' Body ')'

       Body ::= Exp+

        Exp ::= Begin | Var | NExp

      Begin ::= '(' 'begin' Body ')'

        Var ::= '(' 'var' VARIABLE NExp ')'

       NExp ::= Set | If | While | And | Or
              | Apply | VARIABLE
              | Lambda | INTEGER | BOOLEAN | List | Symbol

        Set ::= '(' 'set' VARIABLE NExp ')'

      While ::= '(' 'while' NExp Exp ')'

         If ::= '(' 'if' NExp Exp Exp ')'

        And ::= '(' '&&' NExp NExp NExp* ')'

         Or ::= '(' '||' NExp NExp NExp* ')'

      Apply ::= '(' 'apply' NExp NExp* ')'   // This is the only term that is different!

     Lambda ::= '(' 'lambda' VARIABLE* Exp ')'

       List ::= '(' 'list' NExp* ')'

     Symbol ::= '(' 'sym' SYMBOL ')'

     SYMBOL ::= [a-zA-Z][a-zA-Z0-9]*
   VARIABLE ::= [a-zA-Z][a-zA-Z0-9]*
    INTEGER ::= [0-9]+
    BOOLEAN ::= 'true' | 'false'
*/

public class Preprocessor
{
   public static String preprocess(String expression)
   {
      Tokenizer tokens = new Tokenizer(expression);

      String result = getTree(tokens); // preprocess the tokens

      if ( tokens.hasToken() )  // there shouldn't be any more tokens
         throw new Error("syntax error: unexpected input: "+tokens.peekToken()+"\n"+tokens);

      return result;
   }//preprocess()


   private static String getTree(Tokenizer tokens)
   {
      String result = "";
      String token = tokens.nextToken(); // consume one token
      if ( token.equals("(") )           // look for a parenthesized tree
      {
         result += " (";
         String node = tokens.peekToken(); // one token look ahead
         if ( node.equals("prog")          // check for a non-application term
           || node.equals("begin")
           || node.equals("var")
           || node.equals("set")
           || node.equals("if")
           || node.equals("while")
           || node.equals("&&")
           || node.equals("||")
           || node.equals("lambda")
           || node.equals("list")
           || node.equals("sym") )
         {
            result += " " + tokens.nextToken();
         }
         else if ( node.equals("(") )     // an anonymous application
         {
            result += " apply" + getTree(tokens);
         }
         else  // a named application
         {
            result += " apply " + tokens.nextToken();
         }
         result += getTree(tokens);      // consume first sub tree
         token = tokens.peekToken();     // one token look ahead
         while ( !token.equals(")") )
         {
            result += getTree(tokens);   // consume the sub tree
            token = tokens.peekToken();  // one token look ahead
         }
         tokens.match(")");              // consume the matching ")"
         result += " )";
      }
      else
      {
         result += " " + token;
      }
      return result;
   }//getTree()

}//Preprocessor
