/**
   This program evaluates (interprets) an Abstract Syntax Tree (AST)
   that is defined by the following grammar.

       Prog ::= Exp
              | '(' 'prog' Exp+ Exp ')'

        Exp ::= Begin
              | Var
              | Print
              | BExp
              | AExp
              | INTEGER
              | BOOLEAN
              | VARIABLE

      Begin ::= '(' 'begin' Exp+ Exp ')'

        Var ::= '(' 'var' VARIABLE Exp ')'

      Print ::= '(' 'print' Exp ')'

       BExp ::= '(' '||'  Exp Exp+ ')'
              | '(' '&&'  Exp Exp+ ')'
              | '(' '!'   Exp ')'
              | '(' RelOp Exp Exp ')'
              | '('  EqOp Exp Exp ')'

      RelOp ::= '<' | '>' | '<=' | '>='
       EqOp ::= '==' | '!='

       AExp ::= '(' '+' Exp Exp* ')'
              | '(' '-' Exp Exp? ')'
              | '(' '*' Exp Exp+ ')'
              | '(' '/' Exp Exp  ')'
              | '(' '%' Exp Exp  ')'
              | '(' '^' Exp Exp  ')'

    INTEGER ::= [-][0-9]+
    BOOLEAN ::= 'true' | 'false'
   VARIABLE ::= [a-zA-Z][a-zA-Z0-9]*
*/

public class Evaluate
{
   private static final int DEBUG = 1;

   private static Environment env;  // reference to the environment chain


   public static Value eval(Tree tree) throws EvalException
   {
      return evaluateProg( tree );
   }//eval()


   // Evaluate a prog
   private static Value evaluateProg(Tree tree) throws EvalException
   {
      Value result = null;

      // Instantiate the global environment
      // (it will always be at the end of the
      // environment chain).
      env = new Environment();

      // Check whick kind of Prog we have.
      if ( ! tree.getElement().equals("prog") )
      {
         // Evaluate the single expression.
         result = evaluateExp( tree );
      }
      else
      {
         // Evaluate each Exp in the Prog.
         // Any Var expressions will have the side effect
         // of putting a variable in the environment.
         // Any Print expressions will have the side effect
         // of printing an output.
         // Any other expressions would be pointless!
         for (int i = 0; i < tree.degree()-1; i++)
         {
            evaluateExp( tree.getSubTree(i) );
         }

         // Evaluate the last expression and use its
         // value as the value of the prog expression.
         result = evaluateExp( tree.getSubTree(tree.degree()-1) );
      }

      return result;
   }//evaluateProg()


   // Evaluate an expression
   public static Value evaluateExp(Tree tree) throws EvalException
   {
      Value result = null;

      String node = tree.getElement();

      if ( node.equals("begin") )
         result = evaluateBegin( tree );
      else if ( node.equals("var") )
         result = evaluateVar( tree );
      else if ( node.equals("print") )
         result = evaluatePrint( tree );
      else if ( node.equals("&&") || node.equals("||")
             || node.equals("!") )
         result = evaluateBexp(tree);  // boolean expression
      else if ( node.equals("<")  || node.equals(">")
             || node.equals("<=") || node.equals(">=")
             || node.equals("==") || node.equals("!=") )
         result = evaluateRexp(tree);  // relational operator
      else if ( node.equals("+") || node.equals("-")
             || node.equals("*") || node.equals("/")
             || node.equals("%") || node.equals("^") )
         result = evaluateAexp(tree);  // arithmetic expression
      else if ( tree.degree() == 0 )
      {
         if ( node.equals("true") || node.equals("false") )
            result = new Value( node.equals("true") );
         else if ( node.matches("^[-]*[0-9][0-9]*") )
            result = new Value( Integer.parseInt( node ) );
         else if ( env.defined(node) )  // a variable
            result = env.lookUp( node );
         else  // runtime check
            throw new EvalException("undefined variable: " + node + "\n");
      }
      else
         throw new EvalException("invalid expression: " + tree + "\n");

      return result;
   }//evaluateExp()


   // Evaluate a begin expression
   private static Value evaluateBegin(Tree tree) throws EvalException
   {
      Value result = null;

      // Create a new Environment object chained to (or "nested in")
      // the previous (:outer") environment object.
      Environment previousEnv = env;
      env = new Environment(previousEnv);

      // Evaluate each sub expression in the begin
      // expression (using the new environment chain).
      // The return value of each expression is
      // discarded, so any expression without a
      // side-effect is worthless.
      for (int i = 0; i < tree.degree()-1; i++)
      {
         evaluateExp( tree.getSubTree(i) );
      }

      // Evaluate the last expression and use its
      // value as the value of the begin expression.
      result = evaluateExp( tree.getSubTree(tree.degree()-1) );

      env = previousEnv; // Just before this method returns, we remove from the
                         // chain of Environment objects the local Environment
                         // object we created at the beginning of this method.
                         // The local Environment object becomes a garbage object,
      return result;
   }//evaluateBegin()


   // Evaluate a var expression
   private static Value evaluateVar(Tree tree) throws EvalException
   {
      if ( 2 != tree.degree() )  // runtime check
      {
         throw new EvalException("wrong number of arguments: " + tree + "\n");
      }

      Value result = null;

      // get the variable
      String variable = tree.getSubTree(0).getElement();

      // get, and then evaluate, the expression
      Tree expr = tree.getSubTree(1);
      result = evaluateExp( expr );

      // check if this variable has already been declared
      // in the local environment
      if (! env.definedLocal(variable)) /****Try  env.defined(variable)  instead.****/
      {
         env.add(variable, result);
      }
      else // this variable is already in the environment
      {
         env.update(variable, result);
      }

      if (DEBUG > 0) System.out.println( env + "\n" ); // for debugging purposes

      return result;
   }//evaluateVar()


   // Evaluate a print expression
   private static Value evaluatePrint(Tree tree) throws EvalException
   {
      if ( 1 != tree.degree() )  // runtime check
      {
         throw new EvalException("wrong number of arguments: " + tree + "\n");
      }

      Value result = evaluateExp( tree.getSubTree(0) );

      System.out.println( result + "\n" );

      return result;
   }//evaluatePrint()


   // Evaluate a boolean expression
   private static Value evaluateBexp(Tree tree) throws EvalException
   {
      boolean result = false;

      String node = tree.getElement();

      Value value = evaluateExp( tree.getSubTree(0) );
      if ( ! value.tag.equals(Value.BOOL_TAG) )  // runtime check
      {
         throw new EvalException("not a boolean expression: "
                                  + tree.getSubTree(0) + "\n");
      }
      result = value.valueB;

      if ( node.equals("&&") )
      {
         if ( 2 > tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }

         for (int i = 1; i < tree.degree(); i++)
         {
            if (result)
            {
               value = evaluateExp( tree.getSubTree(i) );
               if ( ! value.tag.equals(Value.BOOL_TAG) )  // runtime check
               {
                  throw new EvalException("not a boolean expression: "
                                           + tree.getSubTree(i) + "\n");
               }
               result = result && value.valueB;
            }
            else  // short circuit the evaluation of '&&'
            {
               result = false;
               break;
            }
         }
      }
      else if ( node.equals("||") )
      {
         if ( 2 > tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }

         for (int i = 1; i < tree.degree(); i++)
         {
            if (! result)
            {
               value = evaluateExp( tree.getSubTree(i) );
               if ( ! value.tag.equals(Value.BOOL_TAG) )  // runtime check
               {
                  throw new EvalException("not a boolean expression: "
                                           + tree.getSubTree(i) + "\n");
               }
               result = result || value.valueB;
            }
            else  // short circuit the evaluation of '||'
            {
               result = true;
               break;
            }
         }
      }
      else if ( node.equals("!") )
      {
         if ( 1 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = ! result;
      }

      return new Value( result );
   }//evaluateBexp()


   // Evaluate a relational expression (which is a kind of boolean expression)
   private static Value evaluateRexp(Tree tree) throws EvalException
   {
      if ( 2 != tree.degree() )  // runtime check
      {
         throw new EvalException("wrong number of arguments: " + tree + "\n");
      }

      boolean result = false;

      String op = tree.getElement();

      Value valueL = evaluateExp( tree.getSubTree(0) );
      if ( ! valueL.tag.equals(Value.INT_TAG) )  // runtime check
      {
         throw new EvalException("not a integer expression: "
                                  + tree.getSubTree(0) + "\n");
      }

      Value valueR = evaluateExp( tree.getSubTree(1) );
      if ( ! valueR.tag.equals(Value.INT_TAG) )  // runtime check
      {
         throw new EvalException("not a integer expression: "
                                  + tree.getSubTree(1) + "\n");
      }

      int resultL = valueL.valueI;
      int resultR = valueR.valueI;

      if ( op.equals("<") )
      {
         result = resultL < resultR;
      }
      else if ( op.equals(">") )
      {
         result = resultL > resultR;
      }
      else if ( op.equals("<=") )
      {
         result = resultL <= resultR;
      }
      else if ( op.equals(">=") )
      {
         result = resultL >= resultR;
      }
      else if ( op.equals("==") )
      {
         result = resultL == resultR;
      }
      else if ( op.equals("!=") )
      {
         result = resultL != resultR;
      }

      return new Value( result );
   }//evaluateRexp()


   // Evaluate an arithmetic expression
   private static Value evaluateAexp(Tree tree) throws EvalException
   {
      int result = 0;

      String node = tree.getElement();

      Value valueL = evaluateExp( tree.getSubTree(0) );
      if ( ! valueL.tag.equals(Value.INT_TAG) )  // runtime check
      {
         throw new EvalException("not a integer expression: "
                                  + tree.getSubTree(0) + "\n");
      }
      int resultL = valueL.valueI;
      int resultR = 0;

      Value valueR = null;
      if ( tree.degree() >= 2 )
      {
         valueR = evaluateExp( tree.getSubTree(1) );
         if ( ! valueR.tag.equals(Value.INT_TAG) )  // runtime check
         {
            throw new EvalException("not a integer expression: "
                                     + tree.getSubTree(1) + "\n");
         }
         resultR = valueR.valueI;
      }

      if ( node.equals("+") )
      {
         if ( tree.degree() == 1 )
            result = resultL;
         else
         {
            result = resultL + resultR;

            for (int i = 2; i < tree.degree(); i++)
            {
               Value temp = evaluateExp( tree.getSubTree(i) );
               if ( ! temp.tag.equals(Value.INT_TAG) )  // runtime check
               {
                  throw new EvalException("not a integer expression: "
                                           + tree.getSubTree(i) + "\n");
               }
               result += temp.valueI;
            }
         }
      }
      else if ( node.equals("-") )
      {
         if ( 2 < tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         if ( tree.degree() == 1 )
            result = -resultL;
         else
            result = resultL - resultR;
      }
      else if ( node.equals("*") )
      {
         if ( 1 == tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }

         result = resultL * resultR;

         for (int i = 2; i < tree.degree(); i++)
         {
            Value temp = evaluateExp( tree.getSubTree(i) );
            if ( ! temp.tag.equals(Value.INT_TAG) )  // runtime check
            {
               throw new EvalException("not a integer expression: "
                                        + tree.getSubTree(i) + "\n");
            }
            result *= temp.valueI;
         }
      }
      else if ( node.equals("/") )
      {
         if ( 2 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = resultL / resultR;
      }
      else if ( node.equals("%") )
      {
         if ( 2 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = resultL % resultR;
      }
      else if ( node.equals("^") )
      {
         if ( 2 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = (int)Math.pow(resultL, resultR);
      }

      return new Value( result );
   }//evaluateAexp()
}//Evaluate