
import java.util.List;
import java.util.ArrayList;

/**
   This implementation is based on a sum algebraic data type.

   In this implementation, all of the leaf nodes of a tree are
   "empty tree" nodes. All nodes that hold a data element are
   actually internal nodes of the tree data structure.
*/
public class BinaryTreeV1<E> extends BinaryTree<E>
{
   private E data; // if this field is null, then the tree is empty
   private BinaryTree<E> left;
   private BinaryTree<E> right;
   private int size;

   /**
      Construct an empty BinaryTreeV1.
   */
   public BinaryTreeV1()
   {
      this.data  = null; // an empty binary tree
      this.left  = null;
      this.right = null;
      this.size = 0;
   }


   /**
      Construct a BinaryTreeV1 with just one node.

      @param data  element of data to store in the root node of this BinaryTreeV1
      @throws NullPointerException
   */
   public BinaryTreeV1(E data)
   {
      this(data, new BinaryTreeV1<E>(), new BinaryTreeV1<E>());
   }


   /**
      Construct a BinaryTree from two other binary trees.

      @param data  element of data to store in the root node of this BinaryTreeV1
      @throws NullPointerException
   */
   public BinaryTreeV1(E data, BinaryTreeV1<E> left, BinaryTreeV1<E> right)
   {
      if (null == data)
         throw new NullPointerException("data cannot be null");
      if (null == left)
         throw new NullPointerException("left cannot be null");
      if (null == right)
         throw new NullPointerException("right cannot be null");

      this.data = data;
      this.left = left;
      this.right = right;
      this.size = 1 + left.size() + right.size();
   }


   /**
      Return the data element stored at the root of this BinaryTreeV1.

      @return the root data element of this BinaryTreeV1
      @throws IllegalStateException if this BinaryTree is empty
   */
   public E data()
   {
      if (0 == this.size)
         throw new IllegalStateException("empty BinaryTree");

      return this.data;
   }


   /**
      Return the left binary sub tree.

      @return the left sub tree of this BinaryTreeV1
      @throws IllegalStateException if this BinaryTreeV1 is empty
   */
   public BinaryTree<E> left()
   {
      if (0 == this.size)
         throw new IllegalStateException("empty BinaryTree");

      return this.left;
   }


   /**
      Return the right binary sub tree.

      @return the right sub tree of this BinaryTreeV1
      @throws IllegalStateException if this BinaryTreeV1 is empty
   */
   public BinaryTree<E> right()
   {
      if (0 == this.size)
         throw new IllegalStateException("empty BinaryTree");

      return this.right;
   }


   /**
      Return true if this BinaryTree does not
      contain any nodes.

      @return true if this BinaryTree is empty
   */
   public boolean isEmpty()
   {
      return null == this.data;
   }


   /**
      Return the number of nodes in this BinaryTreeV1.

      @return the number of nodes in this BinaryTreeV1
   */
   public int size()
   {
      return this.size;
   }


   /**
      Do a pre-order traversal of this BinaryTreeV1.

      @return a List of the pre-order traversal of this BinaryTreeV1
   */
   public List<E> preOrder()
   {
      if (0 == this.size)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         result.add( this.data );
         result.addAll( left.preOrder() );  // recursion
         result.addAll( right.preOrder() ); // recursion
         return result;
      }
   }


   /**
      Do an in-order traversal of this BinaryTreeV1.

      @return a List of the in-order traversal of this BinaryTreeV1
   */
   public List<E> inOrder()
   {
      if (0 == this.size)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         result.addAll( left.inOrder() );  // recursion
         result.add( this.data );
         result.addAll( right.inOrder() ); // recursion
         return result;
      }
   }


   /**
      Do a post-order traversal of this BinaryTreeV1.

      @return a List of the post-order traversal of this BinaryTreeV1
   */
   public List<E> postOrder()
   {
      if (0 == this.size)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         result.addAll( left.postOrder() );  // recursion
         result.addAll( right.postOrder() ); // recursion
         result.add( this.data );
         return result;
      }
   }
}
