
import java.util.List;
import java.util.ArrayList;

/**
   This implementation is based on a sum algebraic data type.

   In this implementation, leaf nodes can hold data and the
   "empty tree" node is only used for a top level empty tree.
*/
public class BinaryTreeV2<E> extends BinaryTree<E>
{
   private E data; // if this field is null, then the tree is empty
   private BinaryTree<E> left;
   private BinaryTree<E> right;
   private int size;

   /**
      Construct an empty BinaryTreeV2.
   */
   public BinaryTreeV2()
   {
      this.data  = null; // an empty binary tree
      this.left  = null;
      this.right = null;
      this.size = 0;
   }


   /**
      Construct a BinaryTreeV2 with just one node.

      @param data  element of data to store in the root node of this BinaryTreeV2
      @throws NullPointerException
   */
   public BinaryTreeV2(E data)
   {
      if (null == data)
         throw new NullPointerException("data cannot be null");

      this.data  = data;  // a leaf binary tree
      this.left  = null;
      this.right = null;
      this.size = 1;
   }


   /**
      Construct a BinaryTree from two other binary trees.

      @param data  element of data to store in the root node of this BinaryTreeV2
      @throws NullPointerException
   */
   public BinaryTreeV2(E data, BinaryTreeV2<E> left, BinaryTreeV2<E> right)
   {
      if (null == data)
         throw new NullPointerException("data cannot be null");
      if (null == left)
         throw new NullPointerException("left cannot be null");
      if (null == right)
         throw new NullPointerException("right cannot be null");

      this.data = data;
      this.left  = (0 == left.size)  ? null : left;
      this.right = (0 == right.size) ? null : right;
      this.size = 1 + left.size() + right.size();
   }


   /**
      Return the data element stored at the root of this BinaryTree.

      @return the root data element of this BinaryTree
      @throws IllegalStateException if this BinaryTree is empty
   */
   public E data()
   {
      if (0 == this.size)
         throw new IllegalStateException("empty BinaryTree");

      return this.data;
   }


   /**
      Return the left binary sub tree.

      @return the left sub tree of this BinaryTreeV2
      @throws IllegalStateException if this BinaryTreeV2 is empty
   */
   public BinaryTree<E> left()
   {
      if (0 == this.size)
         throw new IllegalStateException("empty BinaryTree");

      return (null == left) ? new BinaryTreeV2<>() : this.left;
   }


   /**
      Return the right binary sub tree.

      @return the right sub tree of this BinaryTreeV2
      @throws IllegalStateException if this BinaryTreeV2 is empty
   */
   public BinaryTree<E> right()
   {
      if (0 == this.size)
         throw new IllegalStateException("empty BinaryTree");

      return (null == right) ? new BinaryTreeV2<>() : this.right;
   }


   /**
      Return true if this BinaryTree does not
      contain any nodes.

      @return true if this BinaryTree is empty
   */
   public boolean isEmpty()
   {
      return null == this.data;
   }


   /**
      Return the number of nodes in this BinaryTreeV2.

      @return the number of nodes in this BinaryTree
   */
   public int size()
   {
      return this.size;
   }


   /**
      Do a pre-order traversal of this BinaryTreeV2.

      @return a List of the pre-order traversal of this BinaryTreeV2
   */
   public List<E> preOrder()
   {
      if (0 == this.size)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         result.add( this.data );
         if (null!=left)  result.addAll( left.preOrder() );  // recursion
         if (null!=right) result.addAll( right.preOrder() ); // recursion
         return result;
      }
   }


   /**
      Do an in-order traversal of this BinaryTreeV2.

      @return a List of the in-order traversal of this BinaryTreeV2
   */
   public List<E> inOrder()
   {
      if (0 == this.size)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         if (null!=left)  result.addAll( left.inOrder() );  // recursion
         result.add( this.data );
         if (null!=right) result.addAll( right.inOrder() ); // recursion
         return result;
      }
   }


   /**
      Do a post-order traversal of this BinaryTreeV2.

      @return a List of the post-order traversal of this BinaryTreeV2
   */
   public List<E> postOrder()
   {
      if (0 == this.size)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         if (null!=left)  result.addAll( left.postOrder() );  // recursion
         if (null!=right) result.addAll( right.postOrder() ); // recursion
         result.add( this.data );
         return result;
      }
   }
}
