
import java.util.List;
import java.util.ArrayList;

/**
   This implementation uses a private binary tree node class.
*/
public class BinaryTreeV3<E> extends BinaryTree<E>
{
   class Node // binary tree node
   {
      public E data;
      public Node left;
      public Node right;
      public int size;

      public Node(E data, Node left, Node right)
      {
         this.data = data;
         this.left = left;
         this.right = right;
         this.size = 1 + ((null == left)  ? 0 : left.size)
                       + ((null == right) ? 0 : right.size);
      }

      @Override public String toString()
      {
         return (null==left && null==right)? "" + data :
                "(" + data + ", "
                    + ((null ==  left)? "()" :  left.toString()) + ", "  // recursion
                    + ((null == right)? "()" : right.toString()) + ")";  // recursion
      }
   }


   private Node root; // if this field is null, then the tree is empty

   /**
      Construct an empty BinaryTree.
   */
   public BinaryTreeV3()
   {
      this.root = null;
   }


   /**
      Construct a BinaryTreeV3 with just one node.

      @param data  element of data to store in the root node of this BinaryTreeV3
      @throws NullPointerException
   */
   public BinaryTreeV3(E data)
   {
      this.root = new Node(data, null, null);
   }


   /**
      Construct a BinaryTree from two other binary trees.

      @param data  element of data to store in the root node of this BinaryTreeV3
      @throws NullPointerException
   */
   public BinaryTreeV3(E data, BinaryTreeV3<E> left, BinaryTreeV3<E> right)
   {
      if (null == data)
         throw new NullPointerException("data cannot be null");
      if (null == left)
         throw new NullPointerException("left cannot be null");
      if (null == right)
         throw new NullPointerException("right cannot be null");

      this.root = new Node(data, left.root, right.root);
   }


   /**
      Return the data element stored at the root of this BinaryTreeV3.

      @return the root data element of this BinaryTreeV3
      @throws IllegalStateException if this BinaryTreeV3 is empty
   */
   public E data()
   {
      if (null == root)
         throw new IllegalStateException("empty BinaryTree");

      return this.root.data;
   }


   /**
      Return the left binary sub tree.

      @return the left sub tree of this BinaryTreeV3
      @throws IllegalStateException if this BinaryTreeV3 is empty
   */
   public BinaryTree<E> left()
   {
      if (null == root)
         throw new IllegalStateException("empty BinaryTree");

      BinaryTreeV3<E> left = new BinaryTreeV3<>();
      left.root = this.root.left;
      return left;
   }


   /**
      Return the right binary sub tree.

      @return the right sub tree of this BinaryTreeV3
      @throws IllegalStateException if this BinaryTreeV3 is empty
   */
   public BinaryTree<E> right()
   {
      if (null == root)
         throw new IllegalStateException("empty BinaryTree");

      BinaryTreeV3<E> right = new BinaryTreeV3<>();
      right.root = this.root.right;
      return right;
   }


   /**
      Return true if this BinaryTree does not
      contain any nodes.

      @return true if this BinaryTree is empty
   */
   public boolean isEmpty()
   {
      return null == this.root;
   }


   /**
      Return the number of nodes in this BinaryTreeV3.

      @return the number of nodes in this BinaryTreeV3
   */
   public int size()
   {
      return (null == root) ? 0 : root.size;
   }


   /**
      Do a pre-order traversal of this BinaryTreeV3.

      @return a List of the pre-order traversal of this BinaryTreeV3
   */
   public List<E> preOrder()
   {
      return preOrder(root); // call a recursive helper method
   }

   private List<E> preOrder(Node root)
   {
      if (null == root)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         result.add( root.data );
         result.addAll( preOrder(root.left) );  // recursion
         result.addAll( preOrder(root.right) ); // recursion
         return result;
      }
   }


   /**
      Do an in-order traversal of this BinaryTreeV3.

      @return a List of the in-order traversal of this BinaryTreeV3
   */
   public List<E> inOrder()
   {
      return inOrder(root); // call a recursive helper method
   }

   private List<E> inOrder(Node root)
   {
      if (null == root)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         result.addAll( inOrder(root.left) );  // recursion
         result.add( root.data );
         result.addAll( inOrder(root.right) ); // recursion
         return result;
      }
   }


   /**
      Do a post-order traversal of this BinaryTreeV3.

      @return a List of the post-order traversal of this BinaryTreeV3
   */
   public List<E> postOrder()
   {
      return postOrder(root); // call a recursive helper method
   }

   private List<E> postOrder(Node root)
   {
      if (null == root)
      {
         return new ArrayList<>();
      }
      else
      {
         final List<E> result = new ArrayList<>();
         result.addAll( postOrder(root.left) );  // recursion
         result.addAll( postOrder(root.right) ); // recursion
         result.add( root.data );
         return result;
      }
   }
}
