/**
   This program, as distributed, interprets Language_3.
   You are to modify this program so it interprets the
   following modified version of Language_3.

       Prog ::= Exp
              | '(' 'prog' Exp+ Exp ')'

        Exp ::= PostInc
              | PreInc
              | PostDec
              | PreDec
              | Var
              | Print
              | AExp
              | BExp
              | INTEGER
              | BOOLEAN
              | VARIABLE

       PostInc ::= '(' '++'  VARIABLE ')'
       PostDec ::= '(' '--'  VARIABLE ')'
        PreInc ::= '(' '+++' VARIABLE ')'
        PreDec ::= '(' '---' VARIABLE ')'

        Var ::= '(' 'var' VARIABLE Exp ')'

      Print ::= '(' 'print' Exp ')'

       AExp ::= '(' '+' Exp Exp* ')'
              | '(' '-' Exp Exp? ')'
              | '(' '*' Exp Exp+ ')'
              | '(' '/' Exp Exp  ')'
              | '(' '%' Exp Exp  ')'
              | '(' '^' Exp Exp  ')'

       BExp ::= '(' '||'  Exp Exp+ ')'
              | '(' '&&'  Exp Exp+ ')'
              | '(' '!'   Exp ')'
              | '(' RelOp Exp Exp ')'
              | '('  EqOp Exp Exp ')'

      RelOp ::= '<' | '>' | '<=' | '>='
       EqOp ::= '==' | '!='

    INTEGER ::= [0-9]+
    BOOLEAN ::= 'true' | 'false'
   VARIABLE ::= [a-zA-Z][a-zA-Z0-9]*
*/

public class Evaluate_3a
{
   private static final int DEBUG = 1;

   private static Environment globalEnv;  // global environment data structure

   /**
      The methods eval(), evaluateP(), evaluateE(), evaluateA(), evaluateB(),
      and evaluateR() are essentially a post-order traversal of the abstract
      syntax tree.
   */
   public static Value eval(Tree tree) throws EvalException
   {
      globalEnv = new Environment();      // instantiate the global environment

      return evaluateProg( tree );  // evaluate the program expression
   }//evaluate()


   // Evaluate a prog expression
   private static Value evaluateProg(Tree tree) throws EvalException
   {
      Value result = null;

      if ( ! tree.getElement().equals("prog") )
      {
         // Evaluate the single expression.
         result = evaluateE( tree );
      }
      else
      {
         // Evaluate each Exp term in the Prog term.
         // Any Var terms will have the side effect of putting
         // something in the global environment. Any Print terms
         // will have the side effect of printing an output.
         for (int i = 0; i < tree.degree()-1; i++)
         {
            evaluateE( tree.getSubTree(i) );
         }

         // Evaluate the last expression and use its value as the value of the prog.
         result = evaluateE( tree.getSubTree(tree.degree()-1) );
      }

      return result;
   }//evaluateProg()


   // Evaluate a var expression
   private static Value evaluateV(Tree tree) throws EvalException
   {
      if ( 2 != tree.degree() )  // runtime check
      {
         throw new EvalException("wrong number of arguments: " + tree + "\n");
      }

      Value result = null;

      // get the variable
      String variable = tree.getSubTree(0).getElement();

      // get, and then evaluate, the expression
      Tree expr = tree.getSubTree(1);
      result = evaluateE( expr );

      // check if this variable has already been defined
      if ( ! globalEnv.defined(variable) )
      {
         globalEnv.add(variable, result);
      }
      else // this variable is already in the symbol table
      {
         globalEnv.update(variable, result);
      }

      if (DEBUG > 0) System.out.println( globalEnv + "\n" ); // for debugging purposes

      return result;
   }//evaluateV()



   // Evaluate a print expression
   private static Value evaluatePrint(Tree tree) throws EvalException
   {
      if ( 1 != tree.degree() )  // runtime check
      {
         throw new EvalException("wrong number of arguments: " + tree + "\n");
      }

      Value result = evaluateE( tree.getSubTree(0) );

      System.out.println( result + "\n" );

      return result;
   }//print2Value()


   public static Value evaluateE(Tree tree) throws EvalException
   {
      Value result = null;

      String node = tree.getElement();

      if ( tree.degree() == 0 )  // constant term
      {
         if ( node.equals("true") || node.equals("false") )
         {
            result = new Value( node.equals("true") );
         }
         else if ( node.matches("^[0-9][0-9]*") )
         {
            result = new Value( Integer.parseInt( node ) );
         }
         else if ( globalEnv.defined(node) )  // variable
         {
            result = globalEnv.lookUp( node );
         }
         else  // runtime check
         {
            throw new EvalException("undefined variable: " + node + "\n");
         }
      }
      else if ( node.equals("var") )
      {
         result = evaluateV( tree ); // a var expression
      }
      else if ( node.equals("print") )
      {
         result = evaluatePrint( tree ); // a print expression
      }
      else if ( node.equals("+")
             || node.equals("-")
             || node.equals("*")
             || node.equals("/")
             || node.equals("%")
             || node.equals("^") )
      {
         result = evaluateA(tree);  // arithmetic operator
      }
      else if ( node.equals("&&")
             || node.equals("||")
             || node.equals("!") )
      {
         result = evaluateB(tree);  // boolean operator
      }
      else if ( node.equals("<")
             || node.equals(">")
             || node.equals("<=")
             || node.equals(">=")
             || node.equals("==")
             || node.equals("!=") )
      {
         result = evaluateR(tree);  // relational operator
      }
      else
      {
         throw new EvalException("invalid expression: " + AST2infix_3a.ast2infix(tree) + "\n");
      }

      return result;
   }//evaluateE()


   // Evaluate an arithmetic expression
   private static Value evaluateA(Tree tree) throws EvalException
   {
      int result = 0;

      String node = tree.getElement();

      Value valueL = evaluateE( tree.getSubTree(0) );
      if ( ! valueL.tag.equals(Value.INT_TAG) )  // runtime check
      {
         throw new EvalException("not a integer expression: "
                                  + AST2infix_3a.ast2infix(tree.getSubTree(0)) + "\n");
      }
      int resultL = valueL.valueI;
      int resultR = 0;

      Value valueR = null;
      if ( tree.degree() >= 2 )
      {
         valueR = evaluateE( tree.getSubTree(1) );
         if ( ! valueR.tag.equals(Value.INT_TAG) )  // runtime check
         {
            throw new EvalException("not a integer expression: "
                                     + AST2infix_3a.ast2infix(tree.getSubTree(1)) + "\n");
         }
         resultR = valueR.valueI;
      }

      if ( node.equals("+") )
      {
         if ( tree.degree() == 1 )
            result = resultL;
         else
         {
            result = resultL + resultR;

            for (int i = 2; i < tree.degree(); i++)
            {
               Value temp = evaluateE( tree.getSubTree(i) );
               if ( ! temp.tag.equals(Value.INT_TAG) )  // runtime check
               {
                  throw new EvalException("not a integer expression: "
                                           + AST2infix_3a.ast2infix(tree.getSubTree(i)) + "\n");
               }
               result += temp.valueI;
            }
         }
      }
      else if ( node.equals("-") )
      {
         if ( 2 < tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         if ( tree.degree() == 1 )
            result = -resultL;
         else
            result = resultL - resultR;
      }
      else if ( node.equals("*") )
      {
         if ( 1 == tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }

         result = resultL * resultR;

         for (int i = 2; i < tree.degree(); i++)
         {
            Value temp = evaluateE( tree.getSubTree(i) );
            if ( ! temp.tag.equals(Value.INT_TAG) )  // runtime check
            {
               throw new EvalException("not a integer expression: "
                                        + AST2infix_3a.ast2infix(tree.getSubTree(i)) + "\n");
            }
            result *= temp.valueI;
         }
      }
      else if ( node.equals("/") )
      {
         if ( 2 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = resultL / resultR;
      }
      else if ( node.equals("%") )
      {
         if ( 2 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = resultL % resultR;
      }
      else if ( node.equals("^") )
      {
         if ( 2 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = (int)Math.pow(resultL, resultR);
      }

      return new Value( result );
   }//evaluateA()


   // Evaluate a boolean expression
   private static Value evaluateB(Tree tree) throws EvalException
   {
      boolean result = false;

      String node = tree.getElement();

      Value value = evaluateE( tree.getSubTree(0) );
      if ( ! value.tag.equals(Value.BOOL_TAG) )  // runtime check
      {
         throw new EvalException("not a boolean expression: "
                                  + AST2infix_3a.ast2infix(tree.getSubTree(0)) + "\n");
      }
      result = value.valueB;

      if ( node.equals("&&") )
      {
         if ( 2 > tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }

         for (int i = 1; i < tree.degree(); i++)
         {
            if (result)
            {
               value = evaluateE( tree.getSubTree(i) );
               if ( ! value.tag.equals(Value.BOOL_TAG) )  // runtime check
               {
                  throw new EvalException("not a boolean expression: "
                                           + AST2infix_3a.ast2infix(tree.getSubTree(i)) + "\n");
               }
               result = result && value.valueB;
            }
            else  // short circuit the evaluation of '&&'
            {
               result = false;
               break;
            }
         }
      }
      else if ( node.equals("||") )
      {
         if ( 2 > tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }

         for (int i = 1; i < tree.degree(); i++)
         {
            if (! result)
            {
               value = evaluateE( tree.getSubTree(i) );
               if ( ! value.tag.equals(Value.BOOL_TAG) )  // runtime check
               {
                  throw new EvalException("not a boolean expression: "
                                           + AST2infix_3a.ast2infix(tree.getSubTree(i)) + "\n");
               }
               result = result || value.valueB;
            }
            else  // short circuit the evaluation of '||'
            {
               result = true;
               break;
            }
         }
      }
      else if ( node.equals("!") )
      {
         if ( 1 != tree.degree() )  // runtime check
         {
            throw new EvalException("wrong number of arguments: " + tree + "\n");
         }
         result = ! result;
      }

      return new Value( result );
   }//evaluateB()


   // Evaluate a relational expression (which is a kind of boolean expression)
   private static Value evaluateR(Tree tree) throws EvalException
   {
      if ( 2 != tree.degree() )  // runtime check
      {
         throw new EvalException("wrong number of arguments: " + tree + "\n");
      }

      boolean result = false;

      String op = tree.getElement();

      Value valueL = evaluateE( tree.getSubTree(0) );
      if ( ! valueL.tag.equals(Value.INT_TAG) )  // runtime check
      {
         throw new EvalException("not a integer expression: "
                                  + AST2infix_3a.ast2infix(tree.getSubTree(0)) + "\n");
      }

      Value valueR = evaluateE( tree.getSubTree(1) );
      if ( ! valueR.tag.equals(Value.INT_TAG) )  // runtime check
      {
         throw new EvalException("not a integer expression: "
                                  + AST2infix_3a.ast2infix(tree.getSubTree(1)) + "\n");
      }

      int resultL = valueL.valueI;
      int resultR = valueR.valueI;

      if ( op.equals("<") )
      {
         result = resultL < resultR;
      }
      else if ( op.equals(">") )
      {
         result = resultL > resultR;
      }
      else if ( op.equals("<=") )
      {
         result = resultL <= resultR;
      }
      else if ( op.equals(">=") )
      {
         result = resultL >= resultR;
      }
      else if ( op.equals("==") )
      {
         result = resultL == resultR;
      }
      else if ( op.equals("!=") )
      {
         result = resultL != resultR;
      }

      return new Value( result );
   }//evaluateR()
}//Evaluate_3a
