/**
   This program implements an "interpreter" for Language_0.

   This program provides both a REPL (Read Eval Print Loop)
   for the language and a "script file" interpreter.

   This program assumes that you have installed Graphviz, which includes
   the dot.exe program. You can download Graphviz and dot.exe from the
   following URL. Unzip the zip file to your C:\ drive.

     https://graphviz.gitlab.io/_pages/Download/windows/graphviz-2.38.zip
*/

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.List;

public class Language_0
{
   public static void main(String[] args) throws IOException
   {
      if (args.length > 1)
      {
         System.out.println("Usage: Language_0 <script>");
      }
      else if (args.length == 1)
      {
         runFile(args[0]);
      }
      else
      {
         runPrompt();
      }
   }


   private static void runFile(String path) throws IOException
   {
      byte[] bytes = Files.readAllBytes(Paths.get(path));

      String source = new String(bytes, Charset.defaultCharset());

      // Tokenize, parse, and evaluate the source file.
      try
      {
         Tokenizer tokens = new Tokenizer(source);

         // Print the token list.
         System.out.println(tokens);

         Tree tree = ParseTree.buildTree(tokens);

         // Print the Tree as an S-expression
         System.out.println( tree + "\n" );

         // Pretty-print the tree.
         System.out.println( PrettyPrinter.prettyPrint( tree ) + "\n" );

         // Create dot and png files from the Tree.
         if(true)
         try
         {
            // Create the (empty) dot file.
            String baseName = path;
            java.io.PrintWriter out = new java.io.PrintWriter(
                                         new java.io.File(baseName + ".dot") );
            // Put dot commands into the dot file
            out.println( Tree2dot_ver1.tree2dot(tree) + "\n" );
            out.close();
            // Create a command line for running dot.exe.
            String[] cmd = {"C:\\graphviz-2.38\\release\\bin\\dot.exe",
                            "-Tpng",
                            baseName + ".dot",
                            "-o",
                            baseName + ".png"};
            // Execute the command line.
            java.lang.Runtime.getRuntime().exec(cmd);
         }
         catch (IOException e)
         {
            System.out.println( e );
         }
      }
      catch (TokenizeException e)
      {
         System.out.println(e);
      }
      catch (ParseException e)
      {
         System.out.println(e);
      }
   }


   private static void runPrompt() throws IOException
   {
      InputStreamReader input = new InputStreamReader(System.in);
      BufferedReader reader = new BufferedReader(input);

      System.out.println("Language 0.");

      // repl (read, eval, print loop)
      for (int i = 1;; i++)
      {
         System.out.print("> ");

         // read
         String source = reader.readLine();

         try
         {
            Tokenizer tokens = new Tokenizer(source);
            if ( ! tokens.hasToken() )
            {
               continue; // skip over empty string
            }
            Tree tree = ParseTree.buildTree(tokens);

            if ( tokens.hasToken() )  // there shouldn't be any more tokens
               throw new ParseException("unexpected input: "
                                      + tokens.peekToken().lexeme
                                      + " at line " + tokens.peekToken().line
                                      + ", position " + tokens.peekToken().position
                                      //+ "\n" + tokens
                                      + "\n");

            // Pretty-print the tree.
            System.out.println( PrettyPrinter.prettyPrint( tree ) + "\n" );

            // Create dot and png files from the Tree.
            if(true)
            try
            {
               // Create the (empty) dot file.
               String baseName = String.format("repl.%02d", i);
               java.io.PrintWriter out = new java.io.PrintWriter(
                                            new java.io.File(baseName + ".dot") );
               // Put dot commands into the dot file
               out.println( Tree2dot_ver1.tree2dot(tree) + "\n" );
               out.close();
               // Create a command line for running dot.exe.
               String[] cmd = {"C:\\graphviz-2.38\\release\\bin\\dot.exe",
                               "-Tpng",
                               baseName + ".dot",
                               "-o",
                               baseName + ".png"};
               // Execute the command line.
               java.lang.Runtime.getRuntime().exec(cmd);
            }
            catch (IOException e)
            {
               System.out.println( e );
            }
         }
         catch (TokenizeException e)
         {
            System.out.println(e);
         }
         catch (ParseException e)
         {
            System.out.println(e);
         }
      }
   }
}
