/*
  This program uses a small "Domain Specific Language" (DSL)
  to solve the problem of instantiating trees. Trees are
  represented by strings in a simple Tree language and a
  "parser" reads these strings and builds the appropriate tree.

                a
              / | \
             /  |  \
            b   c    d
          /  \       |
         /    \      |
        e      f     g
                   / | \
                  /  |  \
                 h   i   j
                 |
                 |
                 k

   This program assumes that you have installed Graphviz, which includes
   the dot.exe program. You can download Graphviz and dot.exe from the
   following URL. Unzip the zip file to your C:\ drive.

     https://graphviz.gitlab.io/_pages/Download/windows/graphviz-2.38.zip
*/

public class Language_0_Example_2
{
   public static void main(String[] args) throws ParseException, TokenizeException
   {
      // The string that represents the above tree.
      String treeLanguage = "(a (b e f) c (d (g (h k) i j)))";

      // Build the Tree data structure from its string representation.
      Tree tree = ParseTree.buildTree( treeLanguage );

      // Print the Tree using its toString() method.
      System.out.println( "Tree expression = " + tree + "\n" );

      // The "pretty printer" prints out the tree in a nice format.
      // In a sense, it reverses the process of parsing (and building)
      // the tree, so a "pretty-printer" is sometimes called an "unparser".
      System.out.println( PrettyPrinter.prettyPrint( tree ) + "\n" );

      // Let us traverse the tree so that we can convince
      // ourselves that it was constructed correctly.
      // We should get "a b e f c d g h k i j".
      System.out.println( " Pre-order traversal = " + Traverse.preOrder(  tree ) + "\n" );
      // We should get "e f b c k h i j g d a".
      System.out.println( "Post-order traversal = " + Traverse.postOrder( tree ) + "\n" );


      // Create dot and png files from the tree.
      if(true)
      try
      {
         // Create the (empty) dot file.
         String baseName = "Example_2";
         java.io.PrintWriter out = new java.io.PrintWriter(
                                      new java.io.File(baseName + ".dot"));
         // Put dot commands into the dot file
         out.println( Tree2dot_ver1.tree2dot(tree) + "\n" );
         out.close();

         // Create a command line for running dot.exe.
         String[] cmd = {"C:\\graphviz-2.38\\release\\bin\\dot.exe",
                         "-Tpng",
                         baseName + ".dot",
                         "-o",
                         baseName + ".png"};
         // Execute the command line.
         java.lang.Runtime.getRuntime().exec(cmd);
      }
      catch (Exception e)
      {
         System.out.println( e );
      }
   }
}
