/**
   This program takes an Abstract Syntax Tree (AST)
   that is defined by the following grammar, and
   converts it into a string that contains a
   C/C++/Java like infix notation.

       Prog ::= Exp
              | '(' 'prog' Exp+ Exp ')'

        Exp ::= Fun
              | Lambda
              | Apply
              | If
              | While
              | Set
              | Var
              | Begin
              | Print
              | AExp
              | BExp
              | INTEGER
              | BOOLEAN
              | VARIABLE

        Fun ::= '(' 'fun' VARIABLE Lambda ')'   // a function declaration

     Lambda ::= '(' 'lambda' VARIABLE* Exp ')'  // formal parameters followed by function body

      Apply ::= '(' 'apply' Exp Exp* ')'        // function value followed by actual parameters

         If ::= '(' 'if' Exp Exp Exp ')'

      While ::= '(' 'while' Exp Exp ')'

        Set ::= '(' 'set' VARIABLE Exp ')'

        Var ::= '(' 'var' VARIABLE Exp ')'

      Begin ::= '(' 'begin' Exp+ Exp ')'

      Print ::= '(' 'print' Exp ')'

       BExp ::= '(' '||'  Exp Exp+ ')'
              | '(' '&&'  Exp Exp+ ')'
              | '(' '!'   Exp ')'
              | '(' RelOp Exp Exp ')'
              | '('  EqOp Exp Exp ')'

      RelOp ::= '<' | '>' | '<=' | '>='
       EqOp ::= '==' | '!='

       AExp ::= '(' '+' Exp Exp* ')'
              | '(' '-' Exp Exp? ')'
              | '(' '*' Exp Exp+ ')'
              | '(' '/' Exp Exp  ')'
              | '(' '%' Exp Exp  ')'
              | '(' '^' Exp Exp  ')'

    INTEGER ::= [-][0-9]+
    BOOLEAN ::= 'true' | 'false'
   VARIABLE ::= [a-zA-Z][a-zA-Z0-9]*
*/

public class AST2infix
{
   public static String ast2infix(Tree tree)
   {
      return convertProg( tree, "", true );
   }//ast2infix()


   // Convert a prog
   public static String convertProg(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      // Check whick kind of Prog we have.
      if ( ! tree.getElement().equals("prog") )
      {
         // Convert the single expression.
         result += convertExp( tree, "", false );
      }
      else
      {
         result += "Program\n{\n";

         // convert each expression
         for (int i = 0; i < tree.degree(); i++)
         {
            result += convertExp( tree.getSubTree(i), "   ", false );

            if ( tree.getSubTree(i).getElement().equals("fun") )
            {  // give function definitions an extra \n
               result += "\n";
            }

            if ( tree.getSubTree(i).getElement().equals("fun")
              || tree.getSubTree(i).getElement().equals("begin")
              || tree.getSubTree(i).getElement().equals("if")
              || tree.getSubTree(i).getElement().equals("while")
              || tree.getSubTree(i).getElement().equals("for") )
            {  // the above expressions do not need a semicolon
               result += "\n";
            }
            else // include a semicolon
               result += ";\n";
         }

         result += "}";
      }

      return result;
   }//convertProg()


   // Convert an expression
   public static String convertExp(Tree tree, String indent, boolean inline)
   {
      String result = "";

      String node = tree.getElement();

      if ( node.equals("fun") )
         result += convertFun( tree, indent, inline );
      else if ( node.equals("lambda") )
         result += convertLambda( tree, indent, inline );
      else if ( node.equals("apply") )
         result += convertApply( tree, indent, inline );
      else if ( node.equals("if") )
         result += convertIf( tree, indent, inline );
      else if ( node.equals("while") )
         result += convertWhile( tree, indent, inline );
      else if ( node.equals("set") )
         result += convertSet( tree, indent, inline );
      else if ( node.equals("var") )
         result += convertVar( tree, indent, inline );
      else if ( node.equals("begin") )
         result += convertBegin( tree, indent, inline );
      else if ( node.equals("print") )
         result += convertPrint( tree, indent, inline );
      else if ( node.equals("&&") || node.equals("||")
             || node.equals("!")
             || node.equals("<")  || node.equals(">")
             || node.equals("<=") || node.equals(">=")
             || node.equals("==") || node.equals("!=")
             || node.equals("+")  || node.equals("-")
             || node.equals("*")  || node.equals("/")
             || node.equals("%")  || node.equals("^") )
         result += convertBRAexp( tree, indent, inline );
      else if ( tree.degree() == 0 )
      {
         result += inline ? node : indent + node;
      }
      else // shouldn't get here; just convert to an S-expression
         result += inline ? tree : indent + tree;

      return result;
   }//convertExp()


   // Convert a function definition.
   public static String convertFun(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      // do the name of the function
      result += tree.getSubTree(0).getElement() + "(";
      // get the number of formal parameters from the lambda expression
      int numberOfParameters = tree.getSubTree(1).degree() - 1;
      // do the list of formal parameters from the lambda expression
      for (int i = 0; i < numberOfParameters; i++)
      {
         String variable = tree.getSubTree(1).getSubTree(i).getElement();
         result += variable;
         if (i < numberOfParameters - 1)
            result += ", ";
      }
      result += ")\n";
      // do the function body from the lambda expression
      if ( tree.getSubTree(1).getSubTree(numberOfParameters).getElement().equals("begin") )
      {
         result += convertExp( tree.getSubTree(1).getSubTree(numberOfParameters), indent, false );
      }
      else // a C/C++/Java type function body
      {
         result += indent + "{\n";
         result += convertExp( tree.getSubTree(1).getSubTree(numberOfParameters), indent+"   ", false );
         if ( tree.getSubTree(1).getSubTree(numberOfParameters).getElement().equals("fun")
           || tree.getSubTree(1).getSubTree(numberOfParameters).getElement().equals("while")
           || tree.getSubTree(1).getSubTree(numberOfParameters).getElement().equals("if")
           || tree.getSubTree(1).getSubTree(numberOfParameters).getElement().equals("for") )
         {  // the above expressions do not need a semicolon
            result += "\n" + indent + "}";
         }
         else  // include a semicolon
            result += ";\n" + indent + "}";
      }

      return result;
   }//convertFun()


   // Convert an anonymous function (a lambda expression)
   public static String convertLambda(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      // get the number of formal parameters from the lambda expression
      int numberOfParameters = tree.degree() - 1;
      // do the list of formal parameters from the lambda expression
      if ( 0 == numberOfParameters )
      {
         result += "()";
      }
      else if ( 1 == numberOfParameters )
      {
         // get the formal parameter
         result += tree.getSubTree(0).getElement();
      }
      else
      {
         result += "(";
         // get the first formal parameter
         result += tree.getSubTree(0).getElement();
         for (int i = 1; i < numberOfParameters; i++)
         {
            String variable = tree.getSubTree(i).getElement();
            result += ", " + variable;
         }
         result += ")";
      }

      // use a Maple like notation
      result += " -> ";

      // get the body of the function
      result += convertExp( tree.getSubTree(numberOfParameters), indent, true );

      return result;
   }//convertLambda()


   // Convert a function call
   public static String convertApply(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      // do the function value
      if ( tree.getSubTree(0).degree() > 0 )  // if the function value is more
         result += "(";                       // complicated than just a name
      result += convertExp( tree.getSubTree(0), indent, true);
      if ( tree.getSubTree(0).degree() > 0 )
         result += ")";
      // do the actual parameters
      result += "(";
      for (int i = 1; i < tree.degree(); i++)
      {
         result += convertExp( tree.getSubTree(i), indent, true );
         if (i < tree.degree() - 1)
            result += ", ";
      }
      result += ")";

      return result;
   }//convertApply()


   // Convert an if-expression
   public static String convertIf(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      result += "if (" + convertExp( tree.getSubTree(0), indent+"       ", true ) + ")\n";

      if ( tree.getSubTree(1).getElement().equals("begin") )
      {
         result += convertExp( tree.getSubTree(1), indent, false );
      }
      else if ( tree.getSubTree(1).getElement().equals("fun")
             || tree.getSubTree(1).getElement().equals("while")
             || tree.getSubTree(1).getElement().equals("if")
             || tree.getSubTree(1).getElement().equals("for") )
      {
         result += convertExp( tree.getSubTree(1), indent+"   ", false );
      }
      else
      {
         result += convertExp( tree.getSubTree(1), indent+"   ", false );
         result += ";";
      }

      result += "\n" + indent + "else\n";

      if ( tree.getSubTree(2).getElement().equals("begin") )
      {
         result += convertExp( tree.getSubTree(2), indent, false );
      }
      else if ( tree.getSubTree(2).getElement().equals("fun")
             || tree.getSubTree(2).getElement().equals("while")
             || tree.getSubTree(2).getElement().equals("if")
             || tree.getSubTree(2).getElement().equals("for") )
      {
         result += convertExp( tree.getSubTree(2), indent+"   ", false );
      }
      else
      {
         result += convertExp( tree.getSubTree(2), indent+"   ", false ) + ";";
      }

      return result;
   }//convertIf()


   // Convert a while-loop expression
   public static String convertWhile(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      result += "while (" + convertExp(tree.getSubTree(0), indent+"       ", true) + ")\n";

      if ( tree.getSubTree(1).getElement().equals("begin") )
      {
         result += convertExp( tree.getSubTree(1), indent, false );
      }
      else if ( tree.getSubTree(1).getElement().equals("fun")
             || tree.getSubTree(1).getElement().equals("while")
             || tree.getSubTree(1).getElement().equals("if")
             || tree.getSubTree(1).getElement().equals("for") )
      {
         result += convertExp( tree.getSubTree(1), indent+"   ", false );
      }
      else
      {
         result += convertExp( tree.getSubTree(1), indent+"   ", false );
         result += ";";
      }

      return result;
   }//convertWhile()


   // Convert a set expression
   public static String convertSet(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      // convert the left-hand-side of the assignment expression
      result += convertExp( tree.getSubTree(0), indent, true );

      result += " = ";

      // convert the right-hand-side of the assignment expression
    //indentation += 3 + tree.getSubTree(0).getElement().length();
      result += convertExp( tree.getSubTree(1), "   "+indent, true );

      return result;
   }//convertSet()


   // Convert a var expression
   public static String convertVar(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      // convert the left-hand-side of the declaration expression
      result += "var " + convertExp( tree.getSubTree(0), indent, true );

      result += " = ";

      // convert the right-hand-side of the declaration expression
    //indentation += 7 + tree.getSubTree(0).getElement().length();
      result += convertExp( tree.getSubTree(1), "     "+indent, true );

      return result;
   }//convertVar()


   // Convert a begin expression
   public static String convertBegin(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

         result += "{\n";

      // process each expression
      for(int i = 0; i < tree.degree(); i++)
      {
         result += convertExp( tree.getSubTree(i), indent + "   ", false );
         if ( tree.getSubTree(i).getElement().equals("begin")
           || tree.getSubTree(i).getElement().equals("fun")
           || tree.getSubTree(i).getElement().equals("while")
           || tree.getSubTree(i).getElement().equals("if")
           || tree.getSubTree(i).getElement().equals("for") )
         {  // the above expressions do not need a semicolon
            result += "\n";
         }
         else
            result += ";\n";
      }

      result += indent + "}";

      return result;
   }//convertBegin()


   // Convert a print expression
   public static String convertPrint(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      result += "print(" + convertExp( tree.getSubTree(0), indent, true ) + ")";

      return result;
   }//convertPrint()


   // Convert a boolean, relational, or arithmetic expression
   public static String convertBRAexp(Tree tree, String indent, boolean inline)
   {
      String result = "";

      if ( ! inline )
         result += indent;

      if ( tree.degree() == 1 )  // unary operator
      {
         if ( tree.getSubTree(0).degree() >= 2 )
         {
            result += tree.getElement() + "(" + convertExp( tree.getSubTree(0), indent, true ) + ")";
         }
         else
         {
            result += tree.getElement() + convertExp( tree.getSubTree(0), indent, true );
         }
      }
      else if ( tree.degree() == 2 )  // binary operator
      {
         int middlePrecedence = precedence( tree.getElement() );
         int leftPrecedence = precedence( tree.getSubTree(0).getElement() );

         if ( leftPrecedence > middlePrecedence )
         {
            result += "(" + convertExp( tree.getSubTree(0), indent, true ) + ")";
         }
         else
         {
            result += convertExp( tree.getSubTree(0), indent, true );
         }

         result += " " + tree.getElement() + " ";

         int rightPrecedence = precedence( tree.getSubTree(1).getElement() );

         if ( rightPrecedence >= middlePrecedence )
         {
            result += "(" + convertExp( tree.getSubTree(1), indent, true ) + ")";
         }
         else
         {
            result += convertExp( tree.getSubTree(1), indent, true );
         }
      }
      else  // operators with 3 or more operands
      {
         int operatorPrecedence = precedence( tree.getElement() );
         int leftPrecedence = precedence( tree.getSubTree(0).getElement() );

         if ( leftPrecedence > operatorPrecedence )
         {
            result += "(" + convertExp( tree.getSubTree(0), indent, true ) + ")";
         }
         else
         {
            result += convertExp( tree.getSubTree(0), indent, true );
         }

         for (int i = 1; i < tree.degree(); i++)
         {
            result += " " + tree.getElement() + " ";

            int rightPrecedence = precedence( tree.getSubTree(i).getElement() );

            if ( rightPrecedence >= operatorPrecedence )
            {
               result += "(" + convertExp( tree.getSubTree(i), indent, true ) + ")";
            }
            else
            {
               result += convertExp( tree.getSubTree(i), indent, true );
            }
         }
      }
      return result;
   }//convertBRAexp()



   // http://www.java-tips.org/java-se-tips/java.lang/what-is-java-operator-precedence.html
   // http://www.cs.princeton.edu/introcs/11precedence/
   private static int precedence(String op)
   {
      int result = 0;  // "highest" precedence (i.e., constants)

      if ( op.equals("++")
        || op.equals("--") )
      {
         result = 1;
      }
      else if ( op.equals("index") )
      {
         result = 1;
      }
      else if ( op.equals("apply") )
      {
         result = 1;
      }
      else if ( op.equals("rand") )
      {
         result = 1;
      }
      else if ( op.equals("print") )
      {
         result = 1;
      }
      else if ( op.equals("!")
             || op.equals("+++")
             || op.equals("---") )
      {
         result = 2;
      }
      else if ( op.equals("^") )
      {
         result = 3;
      }
      else if ( op.equals("*")
             || op.equals("/")
             || op.equals("%") )
      {
         result = 4;
      }
      else if ( op.equals("+")
             || op.equals("-") )
      {
         result = 5;
      }
      else if ( op.equals("<")
             || op.equals(">")
             || op.equals("<=")
             || op.equals(">=") )
      {
         result = 7;
      }
      else if ( op.equals("==")
             || op.equals("!=") )
      {
         result = 8;
      }
      else if ( op.equals("&&") )
      {
         result = 12;
      }
      else if ( op.equals("||") )
      {
         result = 13;
      }
      else if ( op.equals("set") )
      {
         result = 15;
      }
      else if ( op.equals("var") )
      {
         result = 15;
      }
      else if ( op.equals("for") )
      {
         result = 16;
      }
      else if ( op.equals("if") )
      {
         result = 16;
      }
      else if ( op.equals("while") )
      {
         result = 16;
      }
      else if ( op.equals("begin") )
      {
         result = 16;
      }

      return result;
   }//precedence()

}//AST2Infix
