/*
   Parallel search for "aardvark" in the stdin stream.

   This version does I/O using a Scanner wrapped around either
   System.in or a BufferedInputStream wrapped around System.in.
   (The difference between them is that the buffer in System.in
   is fixed in size, but we can set the size of the bufer in the
   BufferedInputStream.)

   Each line returned by the Scanner is passed to a parallel
   task handler. The task handler splits its line and then
   compares each token to "aardvark".
*/

import java.time.Instant;
import java.time.Duration;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.Scanner;

public class SearchParallel_v1_stdin
{
   public static void main(String[] args)
   {
      Scanner scanner = new Scanner(System.in);

      final ExecutorService pool = Executors.newFixedThreadPool(12);

      final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
      System.out.println(ft.format(new Date()) + " Search Parallel ver 1 in stdin");
      Instant start = Instant.now();
      int lineNumber = 0;

      while ( scanner.hasNextLine() ) // this blocks
      {
         ++lineNumber;
         pool.execute( new Task(scanner.nextLine(), lineNumber) );
      }
      scanner.close();
      System.out.printf("%s I/O completed (processed %,d lines).\n",
                        ft.format(new Date()), lineNumber);
      pool.shutdown();
      try
      {
         pool.awaitTermination(5, TimeUnit.MINUTES);
      }
      catch (InterruptedException e)
      {
         pool.shutdownNow();
         Thread.currentThread().interrupt(); // Preserve interrupt status
      }

      Instant stop = Instant.now();
      long time = Duration.between(start, stop).toMillis();
      System.out.println(ft.format(new Date()) + " Pool completed.");
      System.out.printf("elapsed time: %,dms\n", time);
      System.out.println( pool.toString() );
   }
}


// A Task object, when it is called by one of the threads in
// the Executor's thread pool, will filter one line of text.
class Task implements Runnable
{
   private static final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
   private final String oneLine;
   private final int lineNumber;

   public Task(final String oneLine, final int lineNumber)
   {
      this.oneLine = oneLine;
      this.lineNumber = lineNumber;
   }

   public void run()
   {
      // Filter the line of input.
      String[] tokens =  oneLine.trim().split("[\\p{Punct}\\s]+");

      for (String s : tokens)
      {
       //if ( s.equals("aardvark") )
         if ( s.indexOf("aardvark") != -1 )
         {
            System.out.printf("%s line %,d\n", ft.format(new Date()), lineNumber);
         }
      }
   }
}//Task
