/*
   Parallel search for "aardvark" in one file called "data".

   This version does file I/O using a BufferedReader wrapped
   around a FileReader.

   Each line returned by the BufferedReader is passed to a parallel
   task handler. The task handler searches its String sequentially
   for "aardvark".
*/

import java.time.Instant;
import java.time.Duration;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.io.IOException;
import java.io.FileReader;
import java.io.BufferedReader;

public class SearchParallel_v3_File
{
   private static final int BUFFERSIZE = 16 * 8192; // changing this doesn't seem to matter

   public static void main(String[] args) throws IOException
   {
      BufferedReader br = new BufferedReader(new FileReader("data"), BUFFERSIZE);

      final ExecutorService pool = Executors.newFixedThreadPool(12);

      final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
      System.out.println(ft.format(new Date()) + " Search Parallel ver 3 in File");
      Instant start = Instant.now();
      int lineNumber = 0;

      String oneLine = null;
      while ( (oneLine = br.readLine()) != null ) // this blocks
      {
         ++lineNumber;
         pool.execute( new Task(oneLine, lineNumber) );
      }
      br.close();
      System.out.printf("%s I/O completed (processed %,d lines).\n",
                        ft.format(new Date()), lineNumber);
      pool.shutdown();
      try
      {
         pool.awaitTermination(5, TimeUnit.MINUTES);
      }
      catch (InterruptedException e)
      {
         pool.shutdownNow();
         Thread.currentThread().interrupt(); // Preserve interrupt status
      }

      Instant stop = Instant.now();
      long time = Duration.between(start, stop).toMillis();
      System.out.println(ft.format(new Date()) + " Pool completed.");
      System.out.printf("elapsed time: %,dms\n", time);
      System.out.println( pool.toString() );
   }
}


// A Task object, when it is called by one of the threads in
// the Executor's thread pool, will filter one line of text.
class Task implements Runnable
{
   private static final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
   private final String oneLine;
   private final int lineNumber;

   public Task(final String oneLine, final int lineNumber)
   {
      this.oneLine = oneLine;
      this.lineNumber = lineNumber;
   }

   public void run()
   {
      // Filter the line of input.
      for (int i = 0; i < oneLine.length() - 8; ++i)
      {
         if ( 'a' == oneLine.charAt(i)
           && 'a' == oneLine.charAt(i+1)
           && 'r' == oneLine.charAt(i+2)
           && 'd' == oneLine.charAt(i+3)
           && 'v' == oneLine.charAt(i+4)
           && 'a' == oneLine.charAt(i+5)
           && 'r' == oneLine.charAt(i+6)
           && 'k' == oneLine.charAt(i+7) )
         {
            System.out.printf("%s line %,d\n", ft.format(new Date()), lineNumber);
         }
      }
   }
}//Task
