/*
   Parallel search for "aardvark" in the stdin stream.

   This version does its own buffering of the System.in stream
   (but notice that System.in is itself a buffered stream).

   This version splits the input buffer into lines and then
   converts each line into another byte buffer, and passes the
   new byte buffer to a parallel task handler. The task handler
   searches its byte buffer sequentially for "aardvark".

   This version is incorrect in that it will miss any "aardvark"
   matches that span across two input buffers (which is not very
   likely to happen).
*/

import java.time.Instant;
import java.time.Duration;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.io.IOException;

public class SearchParallel_v5_stdin
{
   private static final int BUFFERSIZE = 16 * 8192;

   public static void main(String[] args) throws IOException
   {
      byte[] c = new byte[BUFFERSIZE];
      int n;

      final ExecutorService pool = Executors.newFixedThreadPool(12);

      final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
      System.out.println(ft.format(new Date()) + " Search Parallel ver 5 in stdin");
      Instant start = Instant.now();
      int lineNumber = 0;

      while ( (n = System.in.read(c)) != -1 )
      {
         int offset = 0;
         for (int i = 0; i < n; ++i)
         {
            if ( '\n' == c[i] )
            {
               ++lineNumber;
               byte[] oneLine = new byte[i-offset];
               for (int j = 0; j < i-offset; ++j)
               {
                  oneLine[j] = c[offset + j];
               }
               pool.execute( new Task(oneLine, lineNumber) );
               offset = i + 1;
            }
         }
      }
      System.out.printf("%s I/O completed (processed %,d lines).\n",
                        ft.format(new Date()), lineNumber);

      pool.shutdown();
      try
      {
         pool.awaitTermination(5, TimeUnit.MINUTES);
      }
      catch (InterruptedException e)
      {
         pool.shutdownNow();
         Thread.currentThread().interrupt(); // Preserve interrupt status
      }

      Instant stop = Instant.now();
      long time = Duration.between(start, stop).toMillis();
      System.out.println(ft.format(new Date()) + " Pool completed.");
      System.out.printf("elapsed time: %,dms\n", time);
      System.out.println( pool.toString() );
   }
}


// A Task object, when it is called by one of the threads in
// the Executor's thread pool, will filter one line of text.
class Task implements Runnable
{
   private static final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
   private final byte[] oneLine;
   private final int lineNumber;

   public Task(final byte[] oneLine, final int lineNumber)
   {
      this.oneLine = oneLine;
      this.lineNumber = lineNumber;
   }

   public void run()
   {
      //System.out.println( new String(oneLine) );
      //System.out.println( java.util.Arrays.toString(oneLine) );

      // Filter the line of input.
      for (int i = 0; i < oneLine.length - 8; ++i)
      {
         if ( 'a' == oneLine[i]
           && 'a' == oneLine[i+1]
           && 'r' == oneLine[i+2]
           && 'd' == oneLine[i+3]
           && 'v' == oneLine[i+4]
           && 'a' == oneLine[i+5]
           && 'r' == oneLine[i+6]
           && 'k' == oneLine[i+7] )
         {
            System.out.printf("%s line %,d\n", ft.format(new Date()), lineNumber);
         }
      }
   }
}//Task
