/*
   Sequential search for "aardvark" in one file called "data".

   This version memory maps the data file using MappedByteBuffer
   and the map() method in FileChannel.

   The data file is larger than the 2GB mapping limit, so we need
   to map the file in two segments. We may lose an "aardvark" match
   if it just happens to straddle the 2GB position in the file. This
   isn't too hard to fix.
*/

import java.time.Instant;
import java.time.Duration;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.FileInputStream;
import java.io.RandomAccessFile;
//import java.nio.MappedByteBuffer;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

public class SearchParallel_v6MemoryMappedFile
{
   private static final int BUFFERSIZE = 64 * 8192; //256 * 8192;

   public static void main(String[] args) throws IOException, FileNotFoundException
   {
      final ExecutorService pool = Executors.newFixedThreadPool(12);

      FileChannel fc = new RandomAccessFile("data", "r").getChannel();
      long fileSize = fc.size();

      final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
      System.out.println(ft.format(new Date()) + " Search Parallel in Memory Mapped File");
      Instant start = Instant.now();
      int lineNumber = 1;

      for (long i = 0; i < fileSize; i += BUFFERSIZE)
      {
         // account for the last, partially full, buffer
         long bufferSize = (i + BUFFERSIZE < fileSize) ? BUFFERSIZE : fileSize - i;

         ByteBuffer mbb = fc.map( FileChannel.MapMode.READ_ONLY,
                                  i,
                                 (int)bufferSize );
         pool.execute( new Task(mbb, lineNumber) );
         for (int j = 0; j < mbb.capacity(); ++j)
         {
            if ( '\n' == mbb.get(j) )
            {
               ++lineNumber; // count lines here
            }
         }
      }
      System.out.printf("%s memory mapped file traversed (processed %,d lines).\n",
                        ft.format(new Date()), lineNumber);

      pool.shutdown();
      try
      {
         pool.awaitTermination(5, TimeUnit.MINUTES);
      }
      catch (InterruptedException e)
      {
         pool.shutdownNow();
         Thread.currentThread().interrupt(); // Preserve interrupt status
      }

      Instant stop = Instant.now();
      long time = Duration.between(start, stop).toMillis();
      System.out.println(ft.format(new Date()) + " Pool completed.");
      System.out.printf("elapsed time: %,dms\n", time);
      fc.close();
   }
}


// A Task object, when it is called by one of the threads in
// the Executor's thread pool, will filter one memory mapped
// buffer of text.
class Task implements Runnable
{
   private static final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
   private final ByteBuffer buffer;    // a NIO (memory mapped) buffer
   private int lineNumber;

   public Task(final ByteBuffer buffer, final int lineNumber)
   {
      this.buffer = buffer;
      this.lineNumber = lineNumber;
   }

   public void run()
   {
      // Filter the data in the buffer.
      for (int i = 0; i < buffer.capacity(); ++i)
      {
         if ( '\n' == buffer.get(i) )
         {
            ++lineNumber; // count lines here also
         }
         else if ( i < buffer.capacity() - 8
                && 'a' == buffer.get(i)
                && 'a' == buffer.get(i+1)
                && 'r' == buffer.get(i+2)
                && 'd' == buffer.get(i+3)
                && 'v' == buffer.get(i+4)
                && 'a' == buffer.get(i+5)
                && 'r' == buffer.get(i+6)
                && 'k' == buffer.get(i+7) )
         {
            System.out.printf("%s line %,d\n", ft.format(new Date()), lineNumber);
         }
      }
   }
}//Task
