/*
   Parallel search for "aardvark" in the stdin stream.

   This version does its own buffering of the System.in stream
   (but notice that System.in is itself a buffered stream).

   This version does not split the input buffer into lines. It
   passes the whole input buffer to a parallel task handler.
   But since we need to count lines, the serial part of this
   program does count the lines in the input buffer, after it
   has passed the buffer to a task handler. The task handler
   searches its byte buffer sequentially for "aardvark" and
   also has to count lines in its byte buffer.

   This version is incorrect in that it will miss any "aardvark"
   matches that span across two input buffers (which is not very
   likely to happen).
*/

import java.time.Instant;
import java.time.Duration;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.io.IOException;

public class SearchParallel_v6_stdin
{
   private static final int BUFFERSIZE = 256 * 8192;

   public static void main(String[] args) throws IOException
   {
      byte[] c = new byte[BUFFERSIZE];
      int n;

      final ExecutorService pool = Executors.newFixedThreadPool(2);

      final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
      System.out.println(ft.format(new Date()) + " Search Parallel ver 6 in stdin");
      Instant start = Instant.now();
      int lineNumber = 1;

      while ( (n = System.in.read(c)) != -1 )
      {
         pool.execute( new Task(c, n, lineNumber) );
         for (int i = 0; i < n; ++i)
         {
            if ( '\n' == c[i] )
            {
               ++lineNumber; // count lines here
            }
         }
         c = new byte[BUFFERSIZE];
      }
      System.out.printf("%s I/O completed (processed %,d lines).\n",
                        ft.format(new Date()), lineNumber);

      pool.shutdown();
      try
      {
         pool.awaitTermination(5, TimeUnit.MINUTES);
      }
      catch (InterruptedException e)
      {
         pool.shutdownNow();
         Thread.currentThread().interrupt(); // Preserve interrupt status
      }

      Instant stop = Instant.now();
      long time = Duration.between(start, stop).toMillis();
      System.out.println(ft.format(new Date()) + " Pool completed.");
      System.out.printf("elapsed time: %,dms\n", time);
      System.out.println( pool.toString() );
   }
}


// A Task object, when it is called by one of the threads in
// the Executor's thread pool, will filter one buffer of text.
class Task implements Runnable
{
   private static final SimpleDateFormat ft = new SimpleDateFormat("hh:mm:ss");
   private final byte[] buffer;
   private final int n;
   private int lineNumber;

   public Task(final byte[] buffer, final int n, final int lineNumber)
   {
      this.buffer = buffer;
      this.n = n;
      this.lineNumber = lineNumber;
   }

   public void run()
   {
      // Filter the data in the buffer.
      for (int i = 0; i < n; ++i)
      {
         if ( '\n' == buffer[i] )
         {
            ++lineNumber; // count lines here also
         }
         else if ( i < n - 8
                && 'a' == buffer[i]
                && 'a' == buffer[i+1]
                && 'r' == buffer[i+2]
                && 'd' == buffer[i+3]
                && 'v' == buffer[i+4]
                && 'a' == buffer[i+5]
                && 'r' == buffer[i+6]
                && 'k' == buffer[i+7] )
         {
            System.out.printf("%s line %,d\n", ft.format(new Date()), lineNumber);
         }
      }
   }
}//Task
