#include "GLUTcallbacks.h"

extern Scene *scene;     /* defined in mainMaze.cpp */
extern int global_w;     /* viewport width */
extern int global_h;     /* viewport height */
extern double eye_fov;   /* defined in mainMaze.cpp */
extern double eye_theta;
extern double eye_phi;
extern double eye_pos[3];
extern bool opengl_test; /* defined in mainMaze.cpp */
extern bool bilerp_flag; /* defined in mainMaze.cpp */

static Timer timer1;     /* Used to compute rendering speed in frames per second */

static FrameBuffer fb(window_width, window_height); /* This is the object that the
                                                       software renderer will render
                                                       the Scene object into. */

void display(void)
{
   double time;
   char name[128];

   timer1.startTimer();

   /*** set the view port ***/
   if (opengl_test)
   {  /* openGL rendering mode */
      glViewport(0, 0, global_w, global_h);
   }
   else /* software rendering mode */
   {
      fb.setViewport(0, 0, global_w, global_h);
   }


   /*** set the projection matrix ***/
   if (opengl_test)
   {  /* openGL rendering mode */
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      gluPerspective( eye_fov,  /* field of view in degree */
//                      1.0,      /* fixed aspect ration */
                      global_w/(double)global_h, /* varying aspect ratio */
                      0.1,      /* near plane */
                      700 );    /* far plane */
   }
   else /* software rendering mode */
   {
      scene->setProjMatrix2Identity();

      scene->setPerspectiveProj( eye_fov, /* field of view in degree */
//                                  1.0,     /* fixed aspect ration */
                                  global_w/(double)global_h, /* varying aspect ratio */
                                  0.1,     /* near plane */
                                  700 );   /* far plane */
   }


   /*** set the view matrix ***/
   if (opengl_test)
   {  /* openGL rendering mode */
      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity();

      glRotatef(-eye_phi,   1, 0, 0);  // about x-axis
      glRotatef(-eye_theta, 0, 0, 1);  // about z-axis

      /* translate to get the eye origin in the center of the coordinate system */
      glTranslatef(-eye_pos[0], -eye_pos[1], -eye_pos[2]);
   }
   else /* software rendering mode */
   {
      scene->setViewMatrix2Identity();

      //scene->setViewMatrix(eye_pos, eye_theta, eye_phi);
   }


   /*** render the scene ***/
   if (opengl_test)
   {  /* openGL rendering mode */
      renderSceneOpenGL(scene);
   }
   else /* software rendering mode */
   {
      fb.renderScene2Viewport(scene);
      glDrawPixels(fb.width, fb.height, GL_RGB, GL_UNSIGNED_BYTE, fb.pixel_buffer);
   }


   glFinish(); /* wait while everything gets rendered */

   timer1.stopTimer();
   time = timer1.getTime();
   sprintf(name, "%s: %.4lf s, %.2lf fps", opengl_test ? "GL" : "soft", time, 1.0/time);
   glutSetWindowTitle(name);

   glutSwapBuffers();
   return;
}


void reshape(int w, int h)
{
   global_w = w;  /* used to set the viewport */
   global_h = h;

   display();
   return;
}


void redisplay(void)
{
   glutPostRedisplay();
   return;
}


void keyboard(unsigned char key, int x, int y)
{
   switch (key)
   {
      case 27:
      case 'q':
      case 'Q':
         /* quit the program */
         exit(0);
         break;
      case ' ':
         /* toggle between opengl and sofware rendering */
         opengl_test = !opengl_test;
         redisplay();
         break;
      case 'w':
         moveForward();
         redisplay();
         break;
      case 'a':
         moveLeft();
         redisplay();
         break;
      case 'd':
         moveRight();
         redisplay();
         break;
      case 'z':
         moveBack();
         redisplay();
         break;
      case 'u':
         moveUp();
         redisplay();
         break;
      case 'n':
         moveDown();
         redisplay();
         break;
      case 'b':
         /* toggle bilinear interpolation of the textures */
         bilerp_flag = !bilerp_flag;
         switchTextureFilteringOpenGL(scene);
         if (bilerp_flag)
         {
            fprintf(stderr,"glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR)\n");
         }
         else
         {
            fprintf(stderr, "glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST)\n");
         }
         fflush(stderr);
         redisplay();
      default:
         break;
   }
   return;
}
