#include "GLUTMouseCallbacks.h"

extern Scene *scene;     /* defined in mainMaze.cpp */
extern bool opengl_test; /* defined in mainMaze.cpp */

extern double eye_fov;   /* defined in mainMaze.cpp */
extern double eye_theta;
extern double eye_phi;
extern double eye_pos[3];

static double sight_vector[3];  /* vector in the direction of the eye */
static double up_vector[3];
static double right_vector[3];

static double drag_speed = 0.75;
static double move_speed = 0.3;

static double previous_x = 0.0,
              previous_y = 0.0;

static bool mouseJustEntered = false;


/* GLUT callback function */
void mouseEntry(int state)
{
   if (state == GLUT_ENTERED)
   {
      mouseJustEntered = true;
   }
   return;
}


/* GLUT callback function */
void mouseMotion(int x, int y)
{
   if (mouseJustEntered)
   {
      mouseJustEntered = false;
      previous_x = x;
      previous_y = y;
   }

   if ( abs(x - previous_x) > 20 || abs(y - previous_y) > 20 )
   {
      previous_x = x;
      previous_y = y;
   }

   /* The mouse motion changes the "spherical coordinates" of the eye. */
   eye_theta = eye_theta + drag_speed * (previous_x - x);
   eye_phi   = eye_phi   + drag_speed * (previous_y - y);

   /* for debugging */
   fprintf(stderr, "eye_theta = %f, eye_phi = %f\n", eye_theta, eye_phi);
   fflush(stderr);

   eye_phi = (eye_phi <   0) ?   0 : eye_phi;
   eye_phi = (eye_phi > 180) ? 180 : eye_phi;

   previous_x = x;
   previous_y = y;

   computeUpSightAndRightVectors();

   glutPostRedisplay();
   return;
}


/* this function computes some useful vectors to determine where the
   viewer is facing.  this is used in the move functions, since you
   move in the direction you are facing, for example */
void computeUpSightAndRightVectors(void)
{
   /* compute the up vector */
   up_vector[0] = 0;
   up_vector[1] = 1;
   up_vector[2] = 0;
   rotateAboutX(up_vector, eye_phi);
   rotateAboutZ(up_vector, eye_theta);

   /* now compute the line of sight vector */
   sight_vector[0] = 0;
   sight_vector[1] = 0;
   sight_vector[2] = -1;
   rotateAboutX(sight_vector, eye_phi);
   rotateAboutZ(sight_vector, eye_theta);

   /* now take the cross product between sight_vector
      and up_vector to get the right_vector */
   crossProduct(right_vector, sight_vector, up_vector);

   return;
}


/* Functions to move the viewpoint around.
   These functions are called by the GLUT
   keyboard callback handler. */
void moveForward(void)
{
   /* update the x and y components of the eye position
      moving in the direction of the sight vector */
   eye_pos[0] += sight_vector[0] * move_speed;
   eye_pos[1] += sight_vector[1] * move_speed;
   return;
}

void moveLeft(void)
{
   /* update the x and y components of the eye position
      moving in the opposite direction of the right vector */
   eye_pos[0] += -right_vector[0] * move_speed;
   eye_pos[1] += -right_vector[1] * move_speed;
   return;
}

void moveRight(void)
{
   /* update the x and y components of the eye position
      moving in the direction of the right vector */
   eye_pos[0] += right_vector[0] * move_speed;
   eye_pos[1] += right_vector[1] * move_speed;
   return;
}

void moveBack(void)
{
   /* update the x and y components of the eye position
      moving in the opposite direction of the sight vector */
   eye_pos[0] -= sight_vector[0] * move_speed;
   eye_pos[1] -= sight_vector[1] * move_speed;
   return;
}

void moveUp(void)
{
   /* update the z component of the eye position
      moving in the vertical direction */
   eye_pos[2] += 2;
   return;
}
void moveDown(void)
{
   /* update z component of the eye position
      moving in the vertically downward direction */
   eye_pos[2] -= 2;
   return;
}


// these rotations all assume the right-hand rule,
// so if the vector is pointing towards the
// viewer the positive rotation is counterclockwise
void rotateAboutX(double *vector, double angle)
{
   float new_vector[3];

   // note that the ith component of the vector stays the same
   new_vector[1] = cos(angle * MY_PI/180.0) * vector[1] - sin(angle * MY_PI/180.0) * vector[2];
   new_vector[2] = sin(angle * MY_PI/180.0) * vector[1] + cos(angle * MY_PI/180.0) * vector[2];

   // copy it back to calling vector
   vector[1] = new_vector[1];
   vector[2] = new_vector[2];
   return;
}

/*
void rotateAboutY(double *vector, double angle)
{
   float new_vector[3];

   // note that the jth component of the vector stays the same
   new_vector[0] = cos(angle * MY_PI/180.0) * vector[0] + sin(angle * MY_PI/180.0) * vector[2];
   new_vector[2] = -sin(angle * MY_PI/180.0) * vector[0] + cos(angle * MY_PI/180.0) * vector[2];

   // copy it back to calling vector
   vector[0] = new_vector[0];
   vector[2] = new_vector[2];
   return;
}
*/

void rotateAboutZ(double *vector, double angle)
{
   float new_vector[3];

   // note that the kth component of the vector stays the same
   new_vector[0] = cos(angle * MY_PI/180.0) * vector[0] - sin(angle * MY_PI/180.0) * vector[1];
   new_vector[1] = sin(angle * MY_PI/180.0) * vector[0] + cos(angle * MY_PI/180.0) * vector[1];

   // copy it back to calling vector
   vector[0] = new_vector[0];
   vector[1] = new_vector[1];
   return;
}


void crossProduct(double *res_vector, double *vector1, double *vector2)
{
   res_vector[0] = (vector1[1] * vector2[2]) - (vector1[2] * vector2[1]);
   res_vector[1] = (vector1[2] * vector2[0]) - (vector1[0] * vector2[2]);
   res_vector[2] = (vector1[0] * vector2[1]) - (vector1[1] * vector2[0]);
   return;
}


bool floatEqual(double a, double b)
{
   double diff = a - b;
   /* return true if they are within FLOAT_EPSILON of each other */
   return ((-FLOAT_EPSILON < diff) && (diff < FLOAT_EPSILON));
}
