#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "FrameBuffer.h"

FrameBuffer::FrameBuffer(int _w, int _h)
{
   width  = _w;
   height = _h;
   /* set the default viewport */
   vp_ul_x = 0;
   vp_ul_y = 0;
   vp_lr_x = width;
   vp_lr_y = height;

   /* allocate the pixel buffer */
   pixel_buffer = (unsigned char *)malloc(sizeof(unsigned char) * width * height * 3);
   if (!pixel_buffer)
   {
      fprintf(stderr, "ERROR! Unable to allocate memory for frame buffer\n");
      fflush(stderr);
      exit(-1);
   }
}


FrameBuffer::~FrameBuffer()
{
   if (pixel_buffer)
      free(pixel_buffer);
}

void FrameBuffer::setViewport(int _vp_ul_x, int _vp_ul_y, int _vp_lr_x, int _vp_lr_y)
{
   vp_ul_x = _vp_ul_x;
   vp_ul_y = _vp_ul_y;
   vp_lr_x = _vp_lr_x;
   vp_lr_y = _vp_lr_y;
}


/* get the pixel with coordinates (i,j) relative to the current viewport */
void FrameBuffer::getPixel(int i, int j, unsigned char **c)
{
   *c = pixel_buffer + ((vp_ul_y + i)*width*3 + (vp_ul_x + j)*3);
   return;
}


/* set the pixel with coordinates (i,j) relative to the current viewport */
void FrameBuffer::setPixel(int i, int j, unsigned char *c)
{
   unsigned char *p = pixel_buffer + ((vp_ul_y + i)*width*3 + (vp_ul_x + j)*3);
   p[0] = c[0];
   p[1] = c[1];
   p[2] = c[2];
   return;
}


/* write the viewport to the specified file. */
void FrameBuffer::dumpVP2File(char *filename)
{
   dumpPixels2File(vp_ul_x, vp_ul_y, vp_lr_x, vp_lr_y, filename);
   return;
}


/* write the framebuffer to the specified file. */
void FrameBuffer::dumpFB2File(char *filename)
{
   dumpPixels2File(0, 0, width, height, filename);
   return;
}


/* write a rectangular sub array of pixels to the specified file */
void FrameBuffer::dumpPixels2File(int ul_x, int ul_y, int lr_x, int lr_y, char *filename)
{
   char buf[50];
   FILE *fp;
   int p_width  = lr_x - ul_x;
   int p_height = lr_y - ul_y;
   int i;

   /* need to create the file first */
   fp = fopen(filename, "w");
   if (!fp)
   {
      fprintf(stderr, "ERROR! Could not create file %s\n", filename);
      fflush(stderr);
      exit(-1);
   }
   //fprintf(stderr, "Created file %s\n", filename);

   /* create the PPM header information first */
   sprintf(buf, "P6\n%d %d\n%d\n", p_width, p_height, 255);

   /* write the PPM header information */
   if (fwrite(buf, strlen(buf), 1, fp) != 1)
   {
      fprintf(stderr, "ERROR! Could not write to file %s\n", filename);
      fflush(stderr);
      exit(-1);
   }

   /* write the pixel data to the file */
   for (i = 0; i < p_height; i++)
   {  /* write one row of pixels at a time,
         read them from the bottom row of the data buffer
         up towards the top row */
      if (fwrite(pixel_buffer+((p_height-i-1)*3*p_width), 3 * p_width, 1, fp) != 1)
      {
         fprintf(stderr, "ERROR! Could not write to file %s\n", filename);
         fflush(stderr);
         exit(-1);
      }
   }

   fclose(fp);

   return;
}
