/*

*/

package TimeZoneWebApp;

import java.text.DateFormat;
import java.util.Date;
import java.util.TimeZone;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Scanner;

/**
   This is an example of how we can use a web server to
   implement a very simple web application.

   This application expects to find the name of a city in its
   standard input stream and then it looks up that city in Java's
   built-in list of time zones. If the city is found in the list,
   then a web page containing the current time in that city is
   written to the standard output stream. If the name is not in
   Java's list of time zones, then an error web page is written
   to the standard output stream.

   Notice how this program opens two HTML template files instead
   of having a lot of HTML strings imbedded in the Java code. This
   technique both improves the Java code and also separates the
   designing of the web app's appearance from the writing of the
   web app's logic. You can restyle this app's appearance without
   having to change this Java program. Similarly, you can change
   this Java program's implementation without affecting the web
   app's visual appearance.

   To run this web application, copy this program's package folder
      TimeZoneWebApp
   into the root folder of the web server. Then access this URL.

         http://localhost:8080/TimeZoneWebApp/
*/
public class TimeZoneApp
{
   private static final DateFormat timeFormatter = DateFormat.getTimeInstance();
   private static final String[] timeZoneIds = TimeZone.getAvailableIDs();

   /**

   */
   public static void main(String[] args)
   {
      // Get the URL's query string from System.in.
      String query = null;
      try (final Scanner in = new Scanner(System.in))
      {
         query = in.nextLine();
      }

      // Parse it to find the city name.
      final int cityIndex = query.indexOf("=") + 1;
      final String cityName = query.substring(cityIndex);

      // Look up the city name in Java's built-in list of time zones.
      TimeZone timeZone = null;
      String timeZoneId = null;
      for (int i = 0; i < timeZoneIds.length; ++i)
      {
         timeZoneId = timeZoneIds[i];
         final int timeZoneCityIndex = timeZoneId.lastIndexOf('/') + 1;
         final String timeZoneCity = timeZoneId.substring(timeZoneCityIndex);
         if (timeZoneCity.replace('_', ' ').equals(cityName))
         {
            // Found.
            timeZone = TimeZone.getTimeZone(timeZoneId);
            break;
         }
      }

      // Build the response body.
      final File file;
      // Choose a template file.
      if (timeZone != null) // if the city name was found
      {
         file = new File("TimeZoneWebApp/next.html"); // result template page
      }
      else // if the city name was not found
      {
         file = new File("TimeZoneWebApp/error.html"); // error template page
      }

      // Read in the chosen template file.
      String body = "";
      try (final Scanner inFile = new Scanner(file))
      {
         while (inFile.hasNextLine()) // Read the whole html template page.
         {
             body += inFile.nextLine() + "\n";
         }
      }
      catch (FileNotFoundException e)
      {
         System.err.println("====> ERROR: TimeZoneApp: Cannot find file: " + file);
         System.exit(-1);
      }

      // Replace the name template parameter with the name of the city.
      body = body.replaceFirst("###timeZoneCity###", cityName);

      // Replace the time template parameter with the city's time.
      String timeString = null;
      if (timeZone != null) // if the city name was found
      {
         timeFormatter.setTimeZone(timeZone);
         final Date time = new Date();
         timeString = timeFormatter.format(time);
         body = body.replaceFirst("###cityTimeZone###", timeString);
      }

      // Log the query result.
      System.err.println("=====> INFO: TimeZoneApp: city = " + cityName
                                       + ", time zone id = " + timeZoneId
                                               + ", time = " + timeString);

      // Send the entity body to System.out.
      System.out.print(body);
      System.out.flush();
      System.out.close();
   }
}
