/*
   This client uses Java's "high level" abstractions
   for communicating with an HTTP server, the URL,
   URLConnection, and HttpUrlConnection classes.
   See
      https://docs.oracle.com/en/java/javase/21/docs/api/java.base/java/net/URL.html
      https://docs.oracle.com/en/java/javase/21/docs/api/java.base/java/net/URLConnection.html
      https://docs.oracle.com/en/java/javase/21/docs/api/java.base/java/net/HttpURLConnection.html

   This client takes a URL as an optional command-line argument.
*/

import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;

public class HttpClient_v4
{
   public static void main (String[] args)
   {
      final String resourceURL;
      if (args.length > 0)
      {
         resourceURL = args[0];
      }
      else
      {
         resourceURL = "http://example.com/";
      }

      // Get this client's process id number (PID). This helps
      // to identify the client in TaskManager or TCPView.
      final ProcessHandle handle = ProcessHandle.current();
      final long pid = handle.pid();
      System.out.println("CLIENT: Process ID number (PID): " + pid );


      // Send a request to the server.
      System.out.println("CLIENT: retrieving resource: " + resourceURL );
      try
      {
         final URL url = new URL(resourceURL);
         final URLConnection connection = url.openConnection();

         System.out.println("CLIENT: created connection: " + connection );

         // Get some information about the server's response.
         final long date = connection.getDate();
         System.out.println("CLIENT: URL date = " + new java.util.Date(date));

         final String mimeType = connection.getContentType();
         System.out.println("CLIENT: URL content type = " + mimeType);

         final int contentLength = connection.getContentLength();
         System.out.println("CLIENT: URL content length = " + contentLength);

         final String contentEncoding = connection.getContentEncoding();
         System.out.println("CLIENT: URL encoding = " + contentEncoding);

         // Cast the URLConnection to a HttpURLConnection.
         final HttpURLConnection httpConnection = (HttpURLConnection)connection;

         // Check if the HTTP response code is HTTP_OK (200).
         final int code = httpConnection.getResponseCode();
         final String message = httpConnection.getResponseMessage();
         System.out.println("CLIENT: HTTP Status Code: " + code + " " + message);

         // Java has an abstract notion of "content".
         final Object content = connection.getContent();
         System.out.println("CLIENT: URL content is an instance of: " + content.getClass());
         System.out.println("CLIENT: URL content = " + content);

         // Read just the server's response body.
         final BufferedReader in = new BufferedReader(
                                     new InputStreamReader(
                                       connection.getInputStream(), "latin1"));

         for (String oneLine; (oneLine = in.readLine()) != null; )
         {
            System.out.println(oneLine);
         }
         in.close();
      }
      catch (MalformedURLException e)
      {
         System.out.println(e);
      }
      catch (IOException e)
      {
         System.out.println("CLIENT: Error occurred while retrieving URL.");
         System.out.println(e);
      }
   }
}
