/*
   This server reads all the data sent to it by a
   client and stores that data in a file in this
   server's current directory.

   When the server detects end-of-stream on its input
   from the client, it closes the output stream to the
   file, sends the clients a brief response message,
   and then closes the socket to the client.
*/
import java.net.*;
import java.io.*;

public class UploadServer
{
   public static final int SERVER_PORT = 5000; // Should be above 1023.

   public static void main (String[] args)
   {
      final int portNumber;
      if (args.length > 0)
      {
         portNumber = Integer.parseInt(args[0]);
      }
      else
      {
         portNumber = SERVER_PORT;
      }

      int clientCounter = 0;

      // Get this server's process id number (PID). This helps
      // to identify the server in TaskManager or TCPView.
      final ProcessHandle handle = ProcessHandle.current();
      final long pid = handle.pid();
      System.out.println("SERVER: Process ID number (PID): " + pid );

      // Get the name and IP address of the local host and
      // print them on the console for information purposes.
      try
      {
         final InetAddress address = InetAddress.getLocalHost();
         System.out.println("SERVER: Hostname: " + address.getCanonicalHostName() );
         System.out.println("SERVER: IP address: " +address.getHostAddress() );
         System.out.println("SERVER: Using port no. " + portNumber);
      }
      catch (UnknownHostException e)
      {
         System.out.println("Unable to determine this host's address.");
         System.out.println( e );
      }

      // Create the server's listening socket.
      ServerSocket serverSocket = null;
      try
      {
         serverSocket = new ServerSocket(portNumber); // Steps 1, 2.
         System.out.println("SERVER: online");
      }
      catch (IOException e)
      {
         System.out.println("SERVER: Error creating network connection.");
         e.printStackTrace();
         System.exit(-1);
      }

      while (true) // Serve multiple clients.
      {
         Socket socket = null;
         BufferedInputStream   in = null;
         BufferedOutputStream out = null;

         // Wait for an incoming client request.
         try
         {
            socket = serverSocket.accept(); // Steps 3, 6.

            // At this point, a client connection has been made.
            in = new BufferedInputStream(           // Step 8.
                        socket.getInputStream());   // Step 7.

            out = new BufferedOutputStream(         // Step 8.
                         socket.getOutputStream()); // Step 7.
         }
         catch (IOException e)
         {
            System.out.println("SERVER: Error connecting to client.");
            System.out.println( e );
         }

         ++clientCounter;
         // Get the client's IP address and port and log them to the console.
         final InetAddress clientIP = socket.getInetAddress();
         final int clientPort = socket.getPort();
         System.out.println("SERVER: Client " + clientCounter
                          + ": IP: " +  clientIP.getHostAddress()
                          + ", Port: " + clientPort);

         // Step 8.
         // Read all the client's input data and save it in a file.
         int bytesReceived = 0;
         try
         {
            final BufferedOutputStream file =
                     new BufferedOutputStream(
                         new FileOutputStream(
                             new File("client_" + clientCounter)));

            // Since the streams are buffered, its OK
            // to read and write one byte at a time.
            int oneByte;  // Must be an int because of "eof".
            while ((oneByte = in.read()) != -1) // Read in until "eof".
            {
               ++bytesReceived;
               try
               {
                  file.write(oneByte);
               }
               catch (IOException e)
               {
                  System.out.println("Server: Cannot write data to output file.");
                  System.out.println( e );
               }
            }
            try
            {
               file.flush();
               file.close();
            }
            catch (IOException e)
            {
               System.out.println("Server: Cannot write data to output file.");
               System.out.println( e );
            }

            // Send the client a response.
            final String message = "You are client " + clientCounter
                                 + ": Thank you for your " + bytesReceived
                                 + " bytes of data.\n";
            final byte[] response = message.getBytes("iso-8859-1");
            out.write(response); // Send the response to the server.
            out.flush(); // Make sure that the response is sent.

            System.out.println("SERVER: Client " + clientCounter
                             + ": Done. (Received "
                             + bytesReceived + " bytes of data.)");

            // Step 9.
            // We are done with this client's input.
            socket.close();

            System.out.println("SERVER: Client " + clientCounter
                             + ": Closed socket.");
         }
         catch (IOException e)
         {
            System.out.println("SERVER: Error communicating with client (Client no. " + clientCounter + ")");
            System.out.println( e );
         }
      }// Step 11 (wait for another client connection).
   }
}
