/*
 * Renderer 3. The MIT License.
 * Copyright (c) 2022 rlkraft@pnw.edu
 * See LICENSE for details.
*/

package renderer.scene;

/**
   This {@code Camera} data structure represents a camera
   located at the origin, looking down the negative z-axis.
<p>
   This {@code Camera} has associated to it a "view volume"
   that determines what part of space the camera "sees" when
   we use the camera to take a picture (that is, when we
   render a {@link Scene}).
<p>
   This {@code Camera} can "take a picture" two ways, using
   a perspective projection or a parallel (orthographic)
   projection. Each way of taking a picture has a different
   shape for its view volume.
*/
public final class Camera
{
   // Choose either perspective or parallel projection.
   public final boolean perspective;

   /**
      A private {@code Camera} constructor for
      use by the static factory methods.
   */
   private Camera(final boolean perspective)
   {
      this.perspective = perspective;
   }


   /**
      This is a static factory method.
      <p>
      Set up this {@code Camera}'s view volume as a perspective projection
      of an infinite view pyramid extending along the negative z-axis.

      @return a new {@code Camera} object with the default perspective parameter
   */
   public static Camera projPerspective()
   {
      return new Camera(true);
   }


   /**
      This is a static factory method.
      <p>
      Set up this {@code Camera}'s view volume as a parallel (orthographic)
      projection of an infinite view parallelepiped extending along the z-axis.

      @return a new {@code Camera} object with the default orthographic parameter
   */
   public static Camera projOrtho()
   {
      return new Camera(false);
   }


   /**
      For debugging.

      @return {@link String} representation of this {@code Camera} object
   */
   @Override
   public String toString()
   {
      String result = "";
      result += "Camera: \n";
      result += "perspective = " + perspective;
      return result;
   }
}
